﻿define(['common/Error', 'controls/Types', 'common/Enums', 'common/Utilites', 'common/Appearence'], function (Error, Types, Enums, Utilites, Appearence) {

    var t = Enums.type,
        message = function (v, t1, t2) {
            return String.format('Cannot convert {0}({1}) to {2}', JSON.stringify(v), t1, t2);
        },
        messageTruncate = function (v, t) {
            return String.format('Truncation of {0} to type {1}', JSON.stringify(v), t);
        },
        warn = function (v, t1, t2) {
            Error.warn(message(v, t1, t2));
        },
        warnTruncate = function (v, t) {
            Error.warn(messageTruncate(v, t));
        };
    var converters = {};
    var format = {};
    format[Enums.type.date] = 'dd-MM-yyyy';
    format[Enums.type.time] = 'dd.HH:mm:ss';
    format[Enums.type.tod] = 'HH:mm:ss';
    format[Enums.type.dt] = format[Enums.type.date] + ' ' + format[Enums.type.tod];
    format[Enums.serverType.date] = 'dd-MM-yyyy';
    format[Enums.serverType.time] = 'dd.HH:mm:ss';
    format[Enums.serverType.tod] = 'HH:mm:ss';
    format[Enums.serverType.time_of_day] = 'HH:mm:ss';
    format[Enums.serverType.dt] = format[Enums.type.date] + ' ' + format[Enums.type.tod];
    format[Enums.serverType.date_and_time] = format[Enums.serverType.dt];
    function ImplicitConvertations() {
    }

    ImplicitConvertations.prototype.getDateTimeDefaultFormat = function () {
        return format[Enums.serverType.dt];
    };

    ImplicitConvertations.prototype.convert = function (value, destType) {
        if (destType === Enums.serverType.auto) {
            return value;
        }

        var tn = value.typeName;
        if (!Types[tn]) {
            Error.warn("Тип " + tn + " не найден");
            return value;
        }
        var type = Types[tn].type;
        switch (type) {
            case t.object:
                if (destType !== tn) {
                    warn(value.value, type, destType);
                }
                return value;
            case t.array:
                if (destType == Enums.serverType.string)
                    return value;
                if (destType !== tn) {
                    warn(value.value, type, destType);
                }
                return value;
            default:
                return converters[type](value, destType);
        }
    }
    converters[t.enum] = function (value, destType) {
        var destTypeClass = Types[destType].type;

        switch (destTypeClass) {
            case t.number: {
                value.value = truncate(value.value, destType);
                break;
            }
            case t.string:
                value.value = Types[value.typeName].description.displayValues[value.value];
                break;
            case t.object:
                warn(value.value, value.typeName, t.object);
                break;
            case t.array:
                warn(value.value, value.typeName, t.array);
                break;
            case t.date:
                warn(value.value, value.typeName, t.date);
                break;
            case t.boolean: {
                if (value.value === 1) {
                    value.value = true;
                } else if (value.value === 0) {
                    value.value = false;
                } else {
                    warn(value.value, value.typeName, t.boolean);
                }

                break;
            }
            case t.enum: {
                valueToEnum(value, destType);
                break;
            }
        }

        return value;
    }

    converters[t.number] = function (value, destType) {
        var destTypeClass = Types[destType].type;

        switch (destTypeClass) {
            case t.number: {
                value.value = truncate(value.value, destType);
                break;
            }
            case t.string:
                switch (destType) {
                    case Enums.serverType.gradient:
                    case Enums.serverType.color:
                        value.value = Appearence.color.numberToARGB(value.value);
                        return;
                    default:
                        value.value = value.value.toString();
                        return;
                }
                break;
            case t.object:
                warn(value.value, value.typeName, t.object);
                break;
            case t.array:
                warn(value.value, value.typeName, t.array);
                break;
            case t.boolean: {
                if (value.value === 1) {
                    value.value = true;
                } else if (value.value === 0) {
                    value.value = false;
                } else {
                    warn(value.value, value.typeName, t.boolean);
                }

                break;
            }
            case t.enum: {
                valueToEnum(value, destType);
                break;
            }
        }

        return value;
    }

    converters[t.date] = function (value, destType) {
        var destTypeClass = Types[destType].type;

        switch (destTypeClass) {
            case t.date:
                switch (destType) {
                    case Enums.serverType.dt:
                    case Enums.serverType.date_and_time:
                    case Enums.serverType.date:
                        value.value = +value.value;
                        return;
                }
            case t.tod:
                var dt = new Date(+value.value);
                var tod = new Date();
                tod.setTime(0);
                tod.setUTCHours(dt.getHours());
                tod.setUTCMinutes(dt.getMinutes());
                tod.setUTCSeconds(dt.getSeconds());
                tod.setUTCMilliseconds(dt.getMilliseconds());
                value.value = tod.getTime();
                return;
            case t.time:
                value.value = Utilites.timeToLocal(+value.value);
                return;
            case t.string:
                switch (destType) {
                    case Enums.serverType.string:
                        value.value = String.format('{0:' + format[value.typeName] + '}', new Date(+value.value));
                        return;
                    case Enums.serverType.gradient:
                    case Enums.serverType.color:
                        value.value = Appearence.color.numberToARGB(value.value);
                        return;
                }
                break;
            case t.boolean:
                if (value.value > 0)
                    value.value = true;
                else
                    value.value = false;
                break;
            default:
                return value;
        }
    }

    converters[t.tod] = function (value, destType) {
        var destTypeClass = Types[destType].type;

        switch (destTypeClass) {
            case t.date:
                switch (destType) {
                    case Enums.serverType.dt:
                    case Enums.serverType.date:
                    case Enums.serverType.date_and_time:
                        value.value = Utilites.dateToUtc(new Date(+value.value)).getTime();
                        return;
                }
            case t.time:
            case t.tod:
                value.value = +value.value;
                break;
            case t.string:
                switch (destType) {
                    case Enums.serverType.gradient:
                    case Enums.serverType.color:
                        value.value = Appearence.color.numberToARGB(value.value);
                        return;
                    default:
                        value.value = String.format('{0:' + format[Enums.serverType.tod] + '}',
                        Utilites.dateToUtc(new Date(+value.value)));
                        break;
                }
                break;
            case t.boolean:
                if (value.value > 0)
                    value.value = true;
                else
                    value.value = false;
                break;
            default:
                value.value = +value.value;
                break;
        }
        return value.value;
    }

    converters[t.time] = function (value, destType) {
        var destTypeClass = Types[destType].type;

        switch (destTypeClass) {
            case t.date:
                value.value = Utilites.dateToUtc(new Date(+value.value)).getTime();
                break;
            case t.tod:
                value.value = (+value.value) % (24 * 60 * 60 * 1000); //отсекаются дни
                break;
            case t.time:
                value.value = +value.value;
                break;
            case t.string:
                switch (destType) {
                    case Enums.serverType.gradient:
                    case Enums.serverType.color:
                        value.value = Appearence.color.numberToARGB(value.value);
                        return;
                    default:
                        value.value = Utilites.formatTimeToString(value.value, format[Enums.serverType.time]);
                        break;
                }
                break;
            case t.boolean:
                if (value.value > 0)
                    value.value = true;
                else
                    value.value = false;
                break;
            default:
                value.value = +value.value;
                break;
        }

        return value.value;
    }

    converters[t.string] = function (value, destType) {
        var destTypeClass = Types[destType].type;
        if (value.typeName == destType)
            return value;
        switch (destTypeClass) {
            case t.number: {
                switch (value.typeName) {
                    case Enums.serverType.gradient:
                    case Enums.serverType.color:
                        value.value = Appearence.color.ARGBToNumber(value.value);
                        break;
                    default: {
                        var n = Number(value.value);
                        if (isNaN(n)) {
                            warn(value.value, value.typeName, destType);
                            value.value = 0;
                        } else {
                            n = truncate(n, destType);
                            value.value = n;
                        }
                    }
                    break;
                }
                break;
            }
            case t.string: {
                switch (destType) {
                    case Enums.serverType.gradient:
                        break;
                    case Enums.serverType.color:
                        value.value = Appearence.color.toCssColor(value.value);
                        if (!value.value) {
                            value.value = "#00000000";
                        }
                        break;
                    case Enums.serverType.color_matrix:
                        var color = Appearence.color.ARGBtoObj(value.value);
                        if (!color) {
                            value.value = "1 0 0 0 0 0 1 0 0 0 0 0 1 0 0 0 0 0 1 0"; //если это не цвет, то возвращаем единичную матрицу
                        } else {
                            value.value = color.r / 255 + " 0 0 0 0 0 " + color.g / 255 + " 0 0 0 0 0 " + color.b / 255 + " 0 0 0 0 0 " + color.a / 255 + " 0";
                        }
                        break;
                    case Enums.serverType.string:
                        break; //конвертация из цвета в string
                    default:
                        warn(value.value, destTypeClass, destType);
                        break;
                }
                break;
            }
            case t.object:
                warn(value.value, value.typeName, t.object);
                break;
            case t.array:
                warn(value.value, value.typeName, t.array);
                break;
            case t.date:
            case t.tod:
            case t.time:
                switch (value.typeName) {
                    case Enums.serverType.gradient:
                    case Enums.serverType.color:
                        value.value = Appearence.color.ARGBToNumber(value.value);
                        break;
                    case Enums.serverType.string:
                        value.value = Date.parseString(value.value, format[destType]);
                        break;
                    default:
                        warn(value.value, value.typeName, destTypeClass);
                        break;
                }
                break;
            case t.boolean: {
                if (value.value.toLowerCase() === 'true') {
                    value.value = true;
                } else if (Appearence.color.isColor(value.value)) {
                    var obj = Appearence.color.ARGBtoObj(value.value);
                    if (obj) { //если цвет парсится в ARGB
                        if (obj.a == 255 && obj.r == 255 && obj.g == 255 && obj.b == 255) {
                            value.value = true;                ////если белый непрозрачный        
                        }
                    }
                } else {
                    value.value = false;
                }
                break;
            }
            case t.enum: {
                // try to convert from number representation to string
                var n = Number(value.value),
                    index;

                if (value.value !== '' && !isNaN(n)) {
                    value.value = n;
                } else {
                    if ((index = Types[destType].description.values.indexOf(value.value)) < 0 &&
                        (index = Types[destType].description.displayValues.indexOf(value.value)) < 0) {
                        // value is not exists, just warn user                        
                        warn(value.value, value.typeName, destType);
                    } else {
                        value.value = index;
                    }
                }
                break;
            }
        }

        return value;
    }

    converters[t.boolean] = function (value, destType) {
        var destTypeClass = Types[destType].type;

        switch (destTypeClass) {
            case t.number: {
                value.value ? value.value = 1 : value.value = 0;
                break;
            }
            case t.string:
                switch (destType) {
                    case Enums.serverType.gradient:
                    case Enums.serverType.color:
                        value.value ? value.value = "#ffffffff" : value.value = "#00000000";
                        return;
                    default:
                        value.value = value.value.toString();
                        return;
                }
                break;
            case t.object:
                warn(value.value, value.typeName, t.object);
                break;
            case t.array:
                warn(value.value, value.typeName, t.array);
                break;
            case t.date:
            case t.tod:
            case t.time:
                value.value ? value.value = 1 : value.value = 0;
                break;
            case t.enum: {
                value.value ? value.value = 1 : value.value = 0;
                break;
            }
            case t.boolean: {
                break;
            }
        }

        return value;
    }

    function valueToEnum(value, destType) {
        var index;
        // try to convert from number representation to string
        var stringVal = Types[destType].description.values[value.value];
        if (stringVal !== undefined) {
            return;
        } else if ((index = Types[destType].description.values.indexOf(value.value)) != -1) {
            value.value = index;
        } else if ((index = Types[destType].description.displayValues.indexOf(value.value)) != -1) {
            value.value = index;
        } else {
            warn(value.value, value.typeName, destType);
        }
    }

    function truncate(value, dstType) {
        var desc = Types[dstType].description,
            min = desc.min,
            max = desc.max;
        if (value > max) {
            warnTruncate(value, dstType);
            value = max;
        } else if (value < min) {
            warnTruncate(value, dstType);
            value = min;
        }

        if (desc.isInteger === true && !isInt(value)) {
            warnTruncate(value, dstType);
            value = Math.round(value);
        }

        return value;
    }

    function isInt(n) {
        return n % 1 === 0;
    }

    return new ImplicitConvertations();
});