/** 
	* @class Connection
	* @classdesc Represents an connection object. This object can get a value from source, 
	convert it using convertations, and apply converted value to target.
	
	*/
define(['core/convertations/ConvertationChain', 'common/Enums', 'core/AdapterFactory', 'core/ImplicitConvertations', 'common/Error'],
	function (ConvertationChain, Enums, AdapterFactory, ImplicitConvertations, Error) {
	    /** 
		* Creates an connection object
		* @constructor
		* @this {Connection}
		* @param {object{source, target, convertations{array}}} settings Object, which contains convertation settings and connection operation.
		*/
	    function Connection(settings) {
	        this.settings = settings;
	        this.init();
	    };

	    /** 
		* Initializes an connection object
		* @constructor
		* @this {Connection}
		*/
	    Connection.prototype.init = function () {
	        var sourceCtor = AdapterFactory.getByType(this.settings.source.type);
	        var targetCtor = AdapterFactory.getByType(this.settings.target.type);
	        this.source = new sourceCtor(this.settings.source, Enums.EndpointType.source);
	        this.target = new targetCtor(this.settings.target, Enums.EndpointType.target);
	        if (this.settings.convertations && this.settings.convertations.length > 0) {
	            this.convertationChain = new ConvertationChain(this.settings.convertations);
	        }

	        sourceCtor = null;
	        targetCtor = null;
	    };

        Connection.prototype.update = function (sender, sourceOldValue) {
            var newVal = this.get(sender);

            this.set(sender, newVal, sourceOldValue);
        };

	    Connection.prototype.get = function (sender) {
	        return this.source.get(sender);
	    };

	    Connection.prototype.getTarget = function (sender) {
	        return this.target.get(sender);
	    };

	    /** 
		* Sets a value to target accordingly to operation. Converts value if necessary.
		* @function set	
		* @this {Connection}
		*/
	    Connection.prototype.set = function (sender, property, sourceOldValue) {
            if (this.convertationChain) {
                property.value = this.convertationChain.convert(property.value, sender);
	        } else {
	            // parameters passed by reference and values are modified inside convert() function
	            // any type constraint checks also performed by convert() function
	            if (property && property.value != undefined && property.typeName!=undefined) {
	                var targetPropertyType = this.target.getType(sender);
	                if (targetPropertyType != undefined) {
	                    ImplicitConvertations.convert(property, targetPropertyType);
	                } else {
	                    //Error.debug(String.format('Target property({0}) has not type', this.settings.target.path));
	                }
	            }
            }

	        var changed = false;
	        if (this.backward) {

                if (sender.model !== undefined && sender.model.objectId !== undefined) {
                    if (this.lastValues === undefined)
                        this.lastValues = {};
                    if (this.backward.lastValues === undefined)
                        this.backward.lastValues = {};

                    if (this.lastValues[sender.model.objectId] != property.value) {
                        this.backward.lastValues[sender.model.objectId] = property.value;
                        this.lastValues[sender.model.objectId] = property.value;
                        changed = true;
                    }
                }
	            else if (this.lastValue != property.value) {
	                this.backward.lastValue = property.value;
                    this.lastValue = property.value;
	                changed = true;
	            }
	        } else
                changed = true;

            if (changed) { //     
	            this.target.set(sender, property, this.settings.operation, sourceOldValue);
	        }
	    };

	    Connection.prototype.isOneTime = function () {
	        return this.settings.oneTime;
	    }

	    Connection.prototype.destroy = function () {

	    }

	    return Connection;
	});