﻿// <copyright file="TextEditController.js" company="ИнСАТ">
// ИнСАТ, 2014
// </copyright>
// 


/*
* @class TextEditController base class for controllers of text edit controls (like TextBox, MemoEdit)
*/
define(['when','common/Enums', 'common/Appearence', 'base/TextControlController', 'common/Utilites',
    'js!libs/jquery_maskedinput.js', 'js!libs/keyboard_layouts/russian.js!order'],

function (when, Enums, Appearence, TextControlController, Utilites) {
    //here we will define controller for button

    var TextEditController = TextControlController.extend({

        //says whether we are coping value from control to model
        settingText: null,

        init: function () {
            this._super();

            this.ClassName = 'TextEditController';
            //кэшируем, что не запрашивать из моделли при вводе
            this.multiline = false;
            this.skipValidation = false;
        },

        onAddedToParent: function () {
            this._super();

            if (this.mvc.model.getKeyboardBinding() === Enums.AttachType.Parent) {
                this.onKeyboardBindingChanged();
            }
        },

        onRemovingFromParent: function () {
            this._super();

            if (this.mvc.model.getKeyboardBinding() === Enums.AttachType.Parent) {
                this.onKeyboardBindingChanged();
            }
        },

        onAddedToDOM: function () {
            this._super();

            this._changeControlByMultiline();

            if (this.mvc.model.getKeyboardVisibility() !== Enums.KeyboardVisibilityType.None
                //      && this.actualVisible()
             && this.actualEnabled()) {
                var settings = this.buildKeyboardSettings();
                this.textElement().keyboard(settings);
            }

            //force keyboard repositioning
            //otherwise keyboard will be positioned at top left corner of the page
            var keyboard = this.getKeyboard();
            if (keyboard &&
                keyboard.$keyboard.length &&
                this.mvc.model.getKeyboardVisibility() === Enums.KeyboardVisibilityType.Always) {
                keyboard.reveal();
            }
        },       

        onFontMultilineChanged: function (value) {
            this._super(value);
            this.multiline = this.mvc.model.get(Enums.ParameterRoles.FONT_MULTILINE);
        },
        onIsVisibleChanged: function (value) {
            this._super(value);
            this.applyKeyboardSettings();
        },

        onIsEnabledChanged: function (value) {
            this._super(value);
            this.applyKeyboardSettings();
        },
        _changeControlByMultiline: function () {
            var mask = this.mvc.model.get(Enums.ParameterRoles.MASK);
            if (mask) {
                this.textElement().unmask();
                this.skipValidation = false;
                this.maskObj = null;
            }

            var src = this.textElement(),
                dest,
                value = this.mvc.model.getFontMultiline();
            if (value === true) {
                dest = $('<textarea></textarea>');
            } else {
                dest = $('<input type="text"></input>');
            }

            Utilites.copyAttributes(src, dest);

            src.replaceWith(dest);

            src = dest;
            ////заменяем root-элемент в  agility-объекте
            this.mvc.view.$root = src;
            if (mask) {
                //do not check on every pressed key
                this.skipValidation = this.mvc.model.getKeyboardVisibility() !== Enums.KeyboardVisibilityType.None
             && this.actualVisible()
             && this.actualEnabled();

                this.maskObj = this.textElement().mask(mask, { skipValidation: this.skipValidation });
            }
        },

        onKeyPress: function (event) {
            //Chrome key code - 10
            if (this.multiline && event.ctrlKey &&
                (event.keyCode === 13 || event.keyCode === 10)) {
                //WI 10982
                this.textElement().trigger("blur");
                //this.onChange(event);
            }
            this._super(event);
        },

        textElement: function () {

            return this.mvc.view.$();
        },

        isSettingText: function () {
            return this.settingText;
        },

        setSettingText: function (value) {
            this.settingText = value;
        },

        getKeyboard: function () {
            var keyboard = this.textElement().data('keyboard');

            return keyboard;
        },

        getVisibleKeyboardElement: function () {
            if (this.mvc.model.getKeyboardVisibility() === Enums.KeyboardVisibilityType.None) {
                return null;
            }

            var keyboard = this.textElement().data('keyboard');

            if (keyboard && keyboard.$keyboard.length > 0 && keyboard.isVisible()) {
                return keyboard.$keyboard;
            }

            return null;
        },

        /*
        * @method updateKeyboardPosition used to redraw keyboard when control moved or changes its size
        */
        updateKeyboardPosition: function () {
            var keyboard = this.getKeyboard();
            if (keyboard &&
                keyboard.$keyboard.length &&
                keyboard.isVisible()) {
                keyboard.reveal();
            }
        },

        onActualFontSizeChanged: function (newValue) {

            this._super();

            //adjust size of keyboard
            var keyboardEl = this.getVisibleKeyboardElement();

            if (keyboardEl && keyboardEl.length) {
                keyboardEl.css('font-size', newValue + 'px');
            }
        },

        applyBorder: function () {
            this._super();

            //apply border to keyboard
            var keyboardEl = this.getVisibleKeyboardElement();
            if (keyboardEl && keyboardEl.length > 0) {
                this.setupBorderTo(keyboardEl);
            }
        },

        applyKeyboardSettings: function () {

            this.removeKeyboard();

            if (this.mvc.model.getKeyboardVisibility() !== Enums.KeyboardVisibilityType.None
                && this.actualVisible()
                && this.actualEnabled()) {
                var settings = this.buildKeyboardSettings();
                this.textElement().keyboard(settings);

                //hack. because keyboard.destroy calls keyboard.$keuboard.remove 
                //even if $keyboard.length==0 (not visible currently)
                //so if ONFOCUS then add method 'remove'
                if (this.mvc.model.getKeyboardVisibility() === Enums.KeyboardVisibilityType.OnFocus) {
                    if (!this.getKeyboard().$keyboard.remove) {
                        this.getKeyboard().$keyboard.remove = function () {
                            //do nothing because it is stube 
                        };
                    }
                }
            }
        },

        removeKeyboard: function () {
            var keyboard = this.getKeyboard();
            if (keyboard) {
                keyboard.destroy();
            }
        },

        onBeforeDestroy: function () {
            this.removeKeyboard();
            this._super();
        },

        buildKeyboardSettings: function () {

            var settings;

            switch (this.mvc.model.getKeyboardType()) {
                case Enums.KeyboardType.INTERNATIONAL:
                    settings = this.buildInternationalKeyboardSettings();
                    break;
                case Enums.KeyboardType.English:
                    settings = this.buildEnglishKeyboardSettings();
                    break;
                case Enums.KeyboardType.Russian:
                    settings = this.buildRussianKeyboardSettings();
                    break;
                case Enums.KeyboardType.NumPad:
                    settings = this.buildNumKeyboardSettings();
                    break;
                case Enums.KeyboardType.HEX:
                    settings = this.buildHexKeyboardSettings();
                    break;
                default:
                    throw new Error('Unknown keyboard type ' + this.mvc.model.getKeyboardType());
            }

            if (this.mvc.model.getMaxLength()) {
                settings.maxLength = this.mvc.model.getMaxLength();
            }

            if (Utilites.isTablet() && !this.mvc.model.get(Enums.ParameterRoles.IS_READONLY)) {
                settings.forceOpen = true;
            }

            settings.usePreview = false;
            settings.autoAccept = true;
            settings.tabNavigation = true;
            settings.useCombos = false;
            settings.initialFocus = false;
            settings.initialVisibility = this.mvc.model.getIsVisible() &&
                this.mvc.model.getKeyboardVisibility() === Enums.KeyboardVisibilityType.Always;
            settings.openOn = Appearence.keyboard.getJQueryVisibility(this.mvc.model.getKeyboardVisibility());
            settings.alwaysOpen = this.mvc.model.getKeyboardVisibility() === Enums.KeyboardVisibilityType.Always;

            var position = Appearence.keyboard.getJQueryPosition(this.mvc.model.getKeyboardPosition(), this.getKeyboardBerth());

            settings.position = position;

            settings.beforeVisible = (function (that) {
                return function (e, keyboard, el) {
                    that.beforeKeyboardVisible(e, keyboard, el);
                }
            })(this);

            settings.visible = (function (that) {
                return function (e, keyboard, el) {
                    //убираем readonly, если его ставили 
                    //для убирания клавиатуры планшетной
                    if (!that.mvc.model.get(Enums.ParameterRoles.IS_READONLY)) {
                        that.textElement().removeAttr('readonly');
                    }
                }
            })(this);

            return settings;
        },

        /*
        * @method beforeKeyboardVisible called just before keyboard becomes visible
        */
        beforeKeyboardVisible: function (e, keyboard, el) {
            var fontSize = this.textElement().css('font-size');

            //font size
            keyboard.$keyboard.css('font-size', fontSize);

            //border
            this.setupBorderTo(keyboard.$keyboard);

            //background
            var bgValue = this.mvc.model.getBackgroundColor();
            Appearence.background.apply(keyboard.$keyboard, bgValue);
        },

        getKeyboardBerth: function () {
            switch (this.mvc.model.getKeyboardBinding()) {
                case Enums.AttachType.Control:
                    return null;
                case Enums.AttachType.Parent:
                    return this.mvc.parent() ? this.mvc.parent().view.$() : null;
                default:
                    return null;
            }
        },

        buildInternationalKeyboardSettings: function () {
            var config = {
                layout: 'international'
            };

            return config;
        },

        buildEnglishKeyboardSettings: function () {
            var config = {
                layout: 'qwerty'
            };

            return config;
        },

        buildRussianKeyboardSettings: function () {
            var config = {
                layout: 'russian-qwerty'
            };

            return config;
        },

        buildNumKeyboardSettings: function () {
            var config = {
                layout: 'num',
                restrictInput: this._restrictKeyboard(), // Prevent keys not in the displayed keyboard from being typed in 
                preventPaste: true  // prevent ctrl-v and right click                 
            };

            return config;
        },

        buildHexKeyboardSettings: function () {
            var config = {
                layout: 'custom',
                customLayout: {
                    'default': [
                     'C D E F',
                     '8 9 A B',
                     '4 5 6 7',
                     '0 1 2 3',
                     '{bksp} {a} {c}'
                    ]
                },
                restrictInput: this._restrictKeyboard(), // Prevent keys not in the displayed keyboard from being typed in                
            }
        },

        _restrictKeyboard: function () {
            return this.mvc.model.get(Enums.ParameterRoles.MASK) ? false : true;
        },

        skipFireAllProperty: function (property) {

            if (this._super(property)) {
                return true;
            }

            switch (property) {
                case Enums.ParameterRoles.KEYBOARD_TYPE:
                case Enums.ParameterRoles.LAYOUT:
                    return true;
                default:
                    return false;
            }
        },

        modelPropertyChangedInternal: function (event) {

            this._super(event);

            var value = this.mvc.model.get(event.property);

            switch (event.property) {
                case Enums.ParameterRoles.IS_READONLY:
                    this.onIsReadonlyChanged(value);
                    break;
                case Enums.ParameterRoles.KEYBOARD_VISIBILITY:
                    this.onKeyboardVisibilityChanged(value);
                    break;
                case Enums.ParameterRoles.KEYBOARD_TYPE:
                    this.onKeyboardTypeChanged(value);
                    break;
                case Enums.ParameterRoles.LAYOUT:
                    this.onKeyboardPositionChanged(value);
                    break;
                case Enums.ParameterRoles.ATTACH:
                    this.onKeyboardBindingChanged(value);
                    break;
                case Enums.ParameterRoles.MASK:
                    this.onMaskChanged(value);
                    break;
                case Enums.ParameterRoles.MAX_LENGTH:
                    this.onMaxLengthChanged(value);
                    break;
            }
        },

        onIsReadonlyChanged: function (value) {
            if (value) {
                this.textElement().attr('readonly', 'readonly');

            } else {
                this.textElement().removeAttr('readonly');
            }
        },

        onKeyboardVisibilityChanged: function (value) {
            this.applyKeyboardSettings();
        },

        onKeyboardTypeChanged: function (value) {
            this.applyKeyboardSettings();
        },

        onKeyboardPositionChanged: function (value) {
            this.applyKeyboardSettings();
        },

        onKeyboardBindingChanged: function (value) {
            this.applyKeyboardSettings();
        },

        onMaskChanged: function (value) {
            this.textElement().unmask();

            if (value) {
                this.textElement().mask(value);
            }
        },

        onMaxLengthChanged: function (value) {
            if (value) {
                this.mvc.view.$().attr('maxlength', value);
            }
            else {
                this.mvc.view.$().removeAttr('maxlength');
            }

            this.applyKeyboardSettings();
        },

        onBackgroundColorChanged: function (value) {
            this._super(value);
            this.onShadowChanged();
            var keyboardEl = this.getVisibleKeyboardElement();
            if (keyboardEl && keyboardEl.length > 0) {
                Appearence.background.apply(keyboardEl, value);
            }
        },

        applyTextShadow: function () {
            if (Appearence.color.ARGBtoObj(this.mvc.model.getBackgroundColor()).a == 0) {
                var shadowColor = this.mvc.model.getShadowColor();
            var shadowSize = this.mvc.model.getShadowSize();
            if (Appearence.color.ARGBtoObj(this.mvc.model._data.TextColor.value).a != 0) {
                Appearence.shadow.applyTextShadow(this.textElement(), {
                color: shadowColor,
                size: shadowSize
                });
            } else {
                    this.textElement().css('text-shadow', "");
                }
            }
        },

        allowAct: function() {
            this.setSettingText(true);
            if (this.skipValidation) {
                this.textElement().trigger("blur.mask");
            }
            var value = this.textElement().val();
            this.mvc.model.setText(value);
            this.setSettingText(false);
            return when.resolve();       
        },

        denyAct: function() {
            var val = this.mvc.model.get(Enums.ParameterRoles.TEXT);        
            this.textElement().val(val)
            return when.resolve();
        },
        
        onTextChanged: function (value) {
            //if we are not coping value from control to model then 
            //copy it from model to control
             if (!this.isSettingText()) {
                var maxlength = this.mvc.model.get(Enums.ParameterRoles.MAX_LENGTH);
                if (value.length > maxlength) {
                    value = value.substring(0, maxlength);
                }

                this.mvc.model.set(Enums.ParameterRoles.TEXT, value, true);
                this.displayText();
            }
        },

        fireActualVisibleChanged: function (newValue) {
            this._super(newValue);

            this.applyKeyboardSettings();
        },

        fireActualEnabledChanged: function (newValue) {
            this._super(newValue);

            this.applyKeyboardSettings();
        },

        fireAbsoluteXChanged: function () {
            this._super();

            this.updateKeyboardPosition();
        },

        fireAbsoluteYChanged: function () {
            this._super();

            this.updateKeyboardPosition();
        },

        fireActualWidthChanged: function () {
            this._super();

            this.updateKeyboardPosition();
        },

        fireActualHeightChanged: function () {
            this._super();

            this.updateKeyboardPosition();
        },

        onParentAbsoluteXChanged: function () {
            this._super();

            this.updateKeyboardPosition();
        },

        onParentAbsoluteYChanged: function () {
            this._super();

            this.updateKeyboardPosition();
        },
        //
        // Events
        //

        attachEvents: function () {
            this._super();

            var self = this;

            this.textElement().change(function (event) {
                self.onChange(event);
                self.textElement().trigger("blur");
            });
        },

        onChange: function (event) {
            this.askConfirmation();
        }
    });

    return TextEditController;
});
