﻿// <copyright file="TrendController.js" company="ИнСАТ">
// ИнСАТ, 2014
// </copyright>
// _completedRequest

/*
* @class TrendController controller class for Trend control
*/
define(['when', 'common/Enums', 'common/Error', 'base/ControlController',
    'controls/Trend/modules/BackLayer',
    'controls/Trend/modules/Trendline',
    'controls/Trend/modules/MiniLegend',
    'controls/Trend/modules/CanvasPool',
     'server/sources/TrendVariablesSource',
     'controls/Trend/modules/VerticalCursor',
     'controls/Trend/modules/Alghoritms',
     'helpers/contextResolver',
     'controls/Trend/modules/TrendEventWrapper',
     'controls/Trend/modules/StaticVerticalCursor',
     'controls/Trend/modules/HideMiniLegendCircle'
], function (when, Enums, Error, ControlController, BackLayer, Trendline,
    MiniLegend, CanvasPool, TrendVariablesSource, VerticalCursor, Alghoritms,
    contextResolver, TrendEventWrapper, StaticVerticalCursor,
    HideMiniLegendCircle) {

    var pensRegex = /^Pens(\[\d+\])\./gi;

    var TrendController = ControlController.extend({

        init: function () {
            this._super();
            this.initialized = false;
            this.ClassName = 'TrendController';
            this.canvasPool = new CanvasPool();
            this.graphs = [];
            // in ms
            this._redrawTime = 100;
            this.mark = 0;
            this.rightBorder = 0;
            this.selectedGraphIdx = -1;
            this.isUpdated = false;
            this.isSizeChanged = false;
            this.startYRange = null;
            this.defaultXInterval = 0;
            this.updateTimer = null;
            this.itemIdToGraphId = {};
        },

        updateTrendLines: function () {
            var size = this.backLayer.getGridDimensions(),
                interval = this.backLayer.getInterval(),
                isAutoScale = this.mvc.model.getIsAutoScale();
            var canvas_size = this.linesCanvas.getSize();
            if (isAutoScale) {
                this._scaleAxis();
            }

            this.linesCanvas.beginRender();
            this.linesCanvas.clearRect({
                x: 0,
                y: 0,
                width: canvas_size.width,
                height: canvas_size.height
            });

            this.verticalCursor.update(size.width, this.mark);

            var i, serie, data, lastDataItem, cursors, height, staticCursor;
            var serverTime = this.variablesSource.getServerTime();

            if (this.mvc.model.get(Enums.ParameterRoles.SHOW_SERVER_CURSOR) === true) {
                this.staticVerticalCursor.drawVerticalLine(this.backLayer.getGridDimensions(), serverTime, this.mark);
            }

            for (i = 0; i < this.graphs.length; i++) {
                serie = this.graphs[i];
                if (i === this.selectedGraphIdx && isAutoScale) {
                    serie.updateWithRange(this.linesCanvas, size.width, this.backLayer.getYRange(), interval, this.mark, serverTime);
                } else {
                    serie.update(this.linesCanvas, size.width, this.backLayer.getYRange(), interval, this.mark, serverTime);
                }

                data = serie.dataStorage.getAll();
                if (data.length === 0) {
                    continue;
                }

                if (serie.model.drawIfNoData() == true) {
                    lastDataItem = data[data.length - 1];
                    data = data.concat([{
                        quality: lastDataItem.quality,
                        timepoint: serverTime,
                        value: lastDataItem.value
                    }]);
                }

                if (serie.model.getIsVisible() === true) {
                    cursors = this.verticalCursor.getCursors();
                    if (this.mvc.model.get(Enums.ParameterRoles.SHOW_CURSOR) === true) {
                        for (var j = 0, l = cursors.length; j < l; ++j) {
                            this.verticalCursor.drawCursor(this.backLayer.getGridDimensions(), cursors[j], this.mark, data, this.backLayer.getYRange());
                        }
                    }

                    if (this.mvc.model.get(Enums.ParameterRoles.SHOW_SERVER_CURSOR) === true) {
                        this.staticVerticalCursor.drawCursor(this.backLayer.getGridDimensions(), serverTime, this.mark, data, serie);
                    }
                }

                height = this.linesCanvas.canvas.offsetTop - this.backLayer.canvas.canvas.offsetTop;
                cursors = null;
            };

            this.linesCanvas.endRender();
            this.backLayer.onValueChanged(this.mark);
            this.miniLegend.update();
        },

        _createTrendlines: function () {
            this.graphs = [];
            _.forEach(this.mvc.model.getPens(), function (trendlineOpt) {
                var trendline = new Trendline(this.graphs.length, this, trendlineOpt.value);
                trendline.eventTarget.addListener(Enums.TrendEvents.selectTrendline, this._trendLineSelected, this);
                this.graphs.push(trendline);
            }, this);
        },

        onAddedToDOM: function () {
            this._super();
            this.options = {
                model: this.mvc.model,
                canvasPool: this.canvasPool
            };

            this._createTrendlines();

            if (this.graphs.length > 0) {
                for (var i = 0, l = this.graphs.length; i < l; ++i) {
                    if (this.graphs[i].hasDataSource() && this.selectedGraphIdx == -1) {
                        this.selectedGraphIdx = i;
                    }
                    var data = this.graphs[i].model.getAll();
                    for (var property in data) {
                        //тот уникальный случай, когда newValue события реально используется
                        var prop = data[property];
                        if (property === Enums.ParameterRoles.DATA_SOURCE) {
                            //из модели DataSource придет строкой
                            prop.value = Number(prop.value);
                            prop.sourceTime = this._getNewMark();
                            prop.statusCode = 0;
                        }
                        this.graphs[i].model.firePropertyChanged(property, undefined, prop);
                    }
                }
                if (this.selectedGraphIdx == -1) {
                    this.selectedGraphIdx = 0;
            }
            }
            this._initInternal();
            this.initialized = true;
        },

        _initInternal: function () {
            this._createLayers();
            this.variablesSource = new TrendVariablesSource(this.clientId());
            if (this.graphs.length > 0) {
                this._startupDataExchange();
                this._startDrawing();
            }
        },

        _createLayers: function () {
            this.options.size = {
                width: this.mvc.view.$().width(),
                height: this.mvc.view.$().height()
            };

            this.options.max = this.mark;

            if (this.updateTimer) {
                clearInterval(this.updateTimer);
            }

            // if we are restarting
            this.canvasPool.dispose();
            this.backLayer = new BackLayer(this.options);
            this.linesCanvas = this.canvasPool.requestCanvas();
            this.verticalCursor = new VerticalCursor(this.linesCanvas, this.mvc.model, this.backLayer.canvas, this.backLayer.yAxis);
            this.staticVerticalCursor = new StaticVerticalCursor(this.linesCanvas, this.mvc.model, this.backLayer.canvas, this.backLayer.xAxis);
            this._addHitRegions();
            this._setupLinesCanvas();
            this._createMiniLegend();

            _.forEach(this.canvasPool.canvasPoolArray, function (canvas) {
                this.append(canvas.getNode());
            }, this.mvc.view.$());
            this._onIsEnabledChanged();
        },
        
        _addHitRegions: function () {
            if (this.trendEventWrapper) {
                this.trendEventWrapper.dispose();
            }

            this.trendEventWrapper = new TrendEventWrapper(this, this.linesCanvas);
            this.linesCanvas.addHitRegion(Enums.SwipeDirection.swipeRight, this.trendEventWrapper.onMoveHalfPage.bind(this.trendEventWrapper));
            this.linesCanvas.addHitRegion(Enums.SwipeDirection.swipeLeft, this.trendEventWrapper.onMoveHalfPage.bind(this.trendEventWrapper));
            this.linesCanvas.addHitRegion(Enums.PinchDirectional.pinchIn, this.trendEventWrapper.onPinchIn.bind(this.trendEventWrapper));
            this.linesCanvas.addHitRegion(Enums.PinchDirectional.pinchOut, this.trendEventWrapper.onPinchOut.bind(this.trendEventWrapper));
            this.linesCanvas.addHitRegion(Enums.PinchDirectional.pinchEnd, this.trendEventWrapper.onPinchEnd.bind(this.trendEventWrapper));
            this.linesCanvas.addHitRegion(Enums.PinchDirectional.pinchStart, this.trendEventWrapper.onPinchStart.bind(this.trendEventWrapper));
            this.linesCanvas.addHitRegion(Enums.PinchDirectional.pinchMove, this.trendEventWrapper.onPinchMove.bind(this.trendEventWrapper));
            this.linesCanvas.addHitRegion(Enums.EventType.click, this.trendEventWrapper.onCanvasClick.bind(this.trendEventWrapper));
            this.linesCanvas.addHitRegion(Enums.PanType.panStart, this.trendEventWrapper.onCanvasStartPan.bind(this.trendEventWrapper));
            this.linesCanvas.addHitRegion(Enums.PanType.panEnd, this.trendEventWrapper.onCanvasEndPan.bind(this.trendEventWrapper));
       //     this.linesCanvas.addHitRegion(Enums.EventType.mouseUp, this.trendEventWrapper.onRightMouseUp.bind(this.trendEventWrapper)); //не надо масштабирование Y по правому клику (12303)
       //     this.linesCanvas.addHitRegion(Enums.EventType.mouseDown, this.trendEventWrapper.onRightMouseDown.bind(this.trendEventWrapper));
            this.verticalCursor.backLayerCanvas.addHitRegion(Enums.EventType.mouseUp, this.verticalCursor.onBackLayerMouseUp.bind(this.verticalCursor));
            this.verticalCursor.backLayerCanvas.addHitRegion(Enums.EventType.mouseMove, this.verticalCursor.onBackLayerMouseMove.bind(this.verticalCursor));
            this.verticalCursor.backLayerCanvas.addHitRegion(Enums.EventType.mouseDown, this.verticalCursor.onBackLayerMouseDown.bind(this.verticalCursor));
            this.verticalCursor.backLayerCanvas.addHitRegion(Enums.PanType.panEnd, this.verticalCursor.cleanBackLayer.bind(this.verticalCursor));
        },

        _createMiniLegend: function () {
            this.miniLegendCanvas = this.canvasPool.requestCanvas();
            this.miniLegend = new MiniLegend(this.miniLegendCanvas, this, this.mvc.model.getPens(), this.graphs, this.mvc.model);

            this._positionMinilegendCanvas();
            this.miniLegend.render();
            this.miniLegend.eventTarget.addListener(Enums.TrendEvents.trendlineVisibility, this._updateTrendLineState, this);
            this.miniLegend.eventTarget.addListener(Enums.TrendEvents.selectTrendline, this._trendLineSelected, this);
        },

        _setupLinesCanvas: function () {
            var dims = this.backLayer.getGridDimensions();
            this.graphs.forEach(function (graph) {
                graph.setCanvas(this.linesCanvas);
                graph.size = this.backLayer.getGridDimensions();
            }, this);
            dims.width += this.staticVerticalCursor.options.borderWidth+4;
            this.linesCanvas
                .setSize(dims)
                .setPosition(dims);
        },

        _positionMinilegendCanvas: function () {
            var mlSize = this.miniLegend.getDimensions(),
                gridSize = this.backLayer.getGridDimensions();

            this.miniLegendCanvas
                .setSize(mlSize)
                .setPosition({
                    x: gridSize.x + gridSize.width - mlSize.width,
                    y: gridSize.y
                })
                .setNodeSize({
                    width: mlSize.width + 'px',
                    height: mlSize.height + 'px'
                });
            this.miniLegend.hideMiniLegendCircle.xCenter = mlSize.width -
                this.miniLegend.hideMiniLegendCircle.options.circulRadius;
        },

        _trendLineSelected: function (event) {
            var selectedGraphModel = this.graphs[event.idx].model;
            selectedGraphModel.isSelected = event.selectOnly ? true :
                !selectedGraphModel.isSelected;

            this.graphs.forEach(function (graph) {
                if (graph.model.getId() !== event.idx) {
                    graph.model.isSelected = false;
                }
            }, this);

            if (selectedGraphModel.isSelected) {
                this.selectedGraphIdx = event.idx;
            } else {
                this.selectedGraphIdx = 0;
            }

            this._selectTrendline(selectedGraphModel);
        },

        _selectTrendline: function (selectedGraphModel) {
            this._updateYAxis(selectedGraphModel.getStyleSettings());
        },

        _updateYAxis: function (data) {
            this.backLayer.updateYAxis(data);
            this._setupLinesCanvas();
            this._positionMinilegendCanvas();
        },
       
        _redrawXAxis: function () {
            this.backLayer.xAxis.canvas.beginRender();
            this.backLayer.xAxis.draw();
            this.backLayer.xAxis.canvas.endRender();
        },

        _redrawYAxis: function () {
            this.backLayer.yAxis.canvas.beginRender();
            this.backLayer.yAxis.draw();
            this.backLayer.yAxis.canvas.endRender();
        },

        _updateXAxis: function () {
            if (this.backLayer.xAxis.model.get(Enums.ParameterRoles.X_MAJOR_TICK_COUNT) < 1) {
                return;
            }

            this.backLayer.xAxis._positionMajorTicks();
            this._selectTrendline(this.graphs[this.selectedGraphIdx].model);
        },

        _onYRangeUpdate: function (event) {
            var selectedGraphModel = this.graphs[event.idx].model;
            //selectedGraphModel.isSelected = event.selectOnly ? true :
            //    !selectedGraphModel.isSelected;
            if (selectedGraphModel) {
                this._updateYAxis(selectedGraphModel.getStyleSettings());
            }
        },

        _scaleAxis: function () {
            var selectedGraph = this.graphs[this.selectedGraphIdx];
            if (!selectedGraph.hasTimepoints()) {
                //еще нету данных - нечего масштабировать
                return;
            }
            var minMin = selectedGraph.getMin(),
                maxMax = selectedGraph.getMax(),
                range = this.backLayer.getYRange(),
                factor = this.mvc.model.getYAxisRangeFactor();

            if (minMin == maxMax) {
                var delta = factor * minMin;
                this._updateYAxis({
                    min: minMin - delta,
                    max: maxMax + delta
                });
            } else if (range[0] != minMin || range[1] != maxMax) {
                minMin = range[0] != minMin ? minMin - factor * Math.abs(minMin) : minMin;
                maxMax = range[1] != maxMax ? maxMax + factor * maxMax : maxMax;
                this._updateYAxis({ min: minMin, max: maxMax });
            }
        },

        _updateTrendLineState: function (event) {
            this.graphs[event.idx].model.setIsVisible(event.state);
        },

        _updateMiniLegend: function () {
            this._positionMinilegendCanvas();
        },

        _startupDataExchange: function () {
            var connectionSettings = this.addConnectionsForParameter(Enums.ParameterRoles.TREND_LINES);
            this.mvc.model.serverVariables = this.getServerVarsFromConnectionsSettings(connectionSettings);
            this.variablesSource.addToSubscription(this.mvc.model.serverVariables, this.mvc.model.getId());
            this.subscribeDataSourceChanged();
            this.variablesSource.setSourcesOptions({
                interval: this.mvc.model.get(Enums.ParameterRoles.UPDATE_TIME)
            });

            this.variablesSource.subscribe(this.mvc.model.getId()).then(function () {
                this.variablesSource.initArchive().then(function () {                    
                    this.variablesSource.startTimer();
                    if (this.rightBorder < this.variablesSource.getServerTime()) { //Это нужно сделать только после установки подписки, чтоб иметь актуальное serverTime
                        this.rightBorder = this.variablesSource.getServerTime();
                        this._setMark(this.rightBorder);
                    }
                    this.initialized = true;
                    this.update();
                }.bind(this));
            }.bind(this));
        },

        _setupYAxis: function () {
            this._trendLineSelected({
                idx: this.selectedGraphIdx,
                selectOnly: true
            });

        },

        _startDrawing: function () {
            if (this.graphs.length > 0) {
                this._setupYAxis();

                var handler = function () {
                    if (this.timerId === undefined) {
                        this.timerId = setInterval(function () {
                            if (this.mvc.model.getIsAutoScroll() && (this.rightBorder < this.variablesSource.getServerTime())) {
                                if (this.mvc.model.get(Enums.ParameterRoles.SHIFT) == 0) {
                                    this._setMark(this.variablesSource.getServerTime());
                                } else {
                                    this._setAutoscrollCursorPosition();
                                }
                            }
                            this.updateTrendLines();
                            this._setEndStartArhive();
                        }.bind(this), this._redrawTime);
                    }
                }

                this.variablesSource.subscribeSourceDataReadyOnce(handler, this, this.mvc.model.getId());
            }
        },

        _stopDrawing: function () {
            clearInterval(this.timerId);
            this.timerId = undefined;
        },

        _restartDrawing: function () {
            this._stopDrawing();
            this._createLayers();
            this._startDrawing();
        },

        _saveCurrentYRange: function () {
            var yRange = this.backLayer.getYRange(),
                setting = {
                    min: yRange[0],
                    max: yRange[1]
                }

            return setting;
        },

        _onMajorTickHeightChange: function () {
            if (this.isSizeChanged === false) {
                this.defaultXInterval = this.trendEventWrapper.defaultInterval;
            }
            var savedDefaultYRange = this.trendEventWrapper.defaultYRange,
                trendLineRange = this.trendEventWrapper.defaultTrendlineRange;

            this._createLayers();
            this._updateYAxis(this.currentYRange);
            this.trendEventWrapper.defaultTrendlineRange = trendLineRange;
            this.trendEventWrapper.defaultInterval = this.defaultXInterval;
            this.trendEventWrapper.defaultYRange = savedDefaultYRange;
            this.isSizeChanged = true;
            var selectedGraphModel = this.graphs[this.selectedGraphIdx].model;
            this._updateYAxis(selectedGraphModel.getStyleSettings());
        },

        _setInitialYRange: function () {
            this.trendEventWrapper.defaultYRange = this.backLayer.getYRange();
            this.isSavedYRange = true;
        },


        modelPropertyChangedInternal: function (event) {
            this._super(event);
            if (!this.initialized) {
                //график строим по onAddedToDOM
                //строим по уже готовой модели
                return;
            }
            if (this.isSavedYRange !== true) {
                // сохраняем первичные значения, если мы их не задаем в рантайме через контроллы
                this._setInitialYRange();
            }

            var p = event.property;
            //по необъяснимым причинам,FF иногда проскакивает это условие
            if (pensRegex.test(p) || pensRegex.test(p) === true) {
                var idx = +p.match(/\d+/)[0];
                event.idx = idx;
                event.property = event.property.replace(pensRegex, '');
                switch (event.property) {
                    case Enums.ParameterRoles.MAX_Y:
                    case Enums.ParameterRoles.MIN_Y:
                        //при автомасштабировании оси обновляются по значениям
                        if (!this.mvc.model.getIsAutoScale()) {
                        //    event.idx = idx;
                            this._onYRangeUpdate(event);
                            this._setInitialYRange();
                        }
                        return;
                    case Enums.ParameterRoles.DATA_SOURCE:
                        this.itemIdToGraphId[event.newValue.itemId+'/'+event.newValue.path] = event.idx;
                    //    if (event.sourceTime || event.newValue.typeName == "STRING") {
                            this.onDataSourceChanged(event);
                    //    }
                        break;
                }
            }           
            this.currentYRange = this._saveCurrentYRange();
            switch (p) {
                case Enums.ParameterRoles.CHART_BACKGROUND:
                case Enums.ParameterRoles.GRID_LINE_THICKNESS:
                case Enums.ParameterRoles.GRID_LINE_STROKE_COLOR:
                case Enums.ParameterRoles.GRID_LINE_FILL_COLOR:
                    this._onChartBackgroundChanged(event);
                    break;
                case Enums.ParameterRoles.SHIFT:
                    this._onShiftChanged(event);
                    break;
                case Enums.ParameterRoles.TILL:
                    this.onTillChanged(event.newValue.value);
                    break;
                case Enums.ParameterRoles.AUTOSCROLL:
                    this.onAutoscrollChanged(event);
                    break;
                case Enums.ParameterRoles.INTERVAL:
                    this.onIntervalChanged(event);
                    break;
                case Enums.ParameterRoles.RESAMPLE_INTERVAL:
                    this.onResampleChanged(event);
                    break;
                case Enums.ParameterRoles.INTERVAL_PAGES_TURN:
                    this.onIntervalPagesTurn(event);
                    break;
                case Enums.ParameterRoles.UPDATE_TIME:
                    this.onUpdateTimeChanged(event);
                    break;
                case Enums.ParameterRoles.WIDTH:
                case Enums.ParameterRoles.HEIGHT:
                    this._restartDrawing();
                    break;
                case Enums.ParameterRoles.AUTOSCALE:
                    this.onAutoScaleChanged();
                    break;
                case Enums.ParameterRoles.IS_ENABLED:
                    this._onIsEnabledChanged();
                    break;
                case Enums.ParameterRoles.TICK_COUNT_X:
                case Enums.ParameterRoles.X_MAJOR_TICK_WIDTH:
                    this._updateXAxis();
                    break;
                case Enums.ParameterRoles.SHOW_CURSOR:
                    this.verticalCursor.cleanBackLayer();
                    break;
                case Enums.ParameterRoles.X_MAJOR_TICK_COLOR:
                case Enums.ParameterRoles.X_ARROW_FILL_COLOR:
                case Enums.ParameterRoles.X_ARROW_STROKE_COLOR:
                case Enums.ParameterRoles.X_MINOR_TICK_COLOR:
                case Enums.ParameterRoles.X_AXIS_FILL_COLOR:
                case Enums.ParameterRoles.X_ARROW_STROKE_COLOR:
                    this._redrawXAxis();
                    break;
                case Enums.ParameterRoles.Y_MAJOR_TICK_COLOR:
                case Enums.ParameterRoles.Y_ARROW_FILL_COLOR:
                case Enums.ParameterRoles.Y_AXIS_STROKE_COLOR:
                case Enums.ParameterRoles.Y_MINOR_TICK_COLOR:
                case Enums.ParameterRoles.Y_AXIS_FILL_COLOR:
                case Enums.ParameterRoles.Y_ARROW_STROKE_COLOR:
                    this._redrawYAxis();
                    break;
                case Enums.ParameterRoles.X_MAJOR_TICK_PADDING_LEFT:
                case Enums.ParameterRoles.X_MAJOR_TICK_PADDING_RIGHT:
                case Enums.ParameterRoles.X_MINOR_TICK_HEIGHT:
                case Enums.ParameterRoles.X_MINOR_TICK_COUNT:
                case Enums.ParameterRoles.X_ARROW_WIDTH:
                case Enums.ParameterRoles.X_ARROW_HEIGHT:
                case Enums.ParameterRoles.X_ARROW_MARGIN:
                case Enums.ParameterRoles.X_AXIS_FORMAT:
                case Enums.ParameterRoles.X_AXIS_LABEL:
                case Enums.ParameterRoles.Y_AXIS_LABEL:
                case Enums.ParameterRoles.Y_AXIS_THICKNESS:
                case Enums.ParameterRoles.Y_MAJOR_TICK_COUNT:
                case Enums.ParameterRoles.Y_MAJOR_TICK_WIDTH:
                case Enums.ParameterRoles.Y_MAJOR_TICK_HEIGHT:
                case Enums.ParameterRoles.Y_MAJOR_TICK_PADDING_LEFT:
                case Enums.ParameterRoles.Y_MAJOR_TICK_PADDING_RIGHT:
                case Enums.ParameterRoles.Y_MINOR_TICK_WIDTH:
                case Enums.ParameterRoles.Y_MINOR_TICK_HEIGHT:
                case Enums.ParameterRoles.Y_MINOR_TICK_COUNT:
                case Enums.ParameterRoles.Y_ARROW_WIDTH:
                case Enums.ParameterRoles.Y_ARROW_HEIGHT:
                case Enums.ParameterRoles.Y_ARROW_MARGIN:
                case Enums.ParameterRoles.X_MAJOR_TICK_HEIGHT:
                case Enums.ParameterRoles.X_MAJOR_TICK_PADDING_TOP:
                case Enums.ParameterRoles.X_MAJOR_TICK_PADDING_BOTTOM:
                case Enums.ParameterRoles.TITLE:
                    this._onMajorTickHeightChange();
                    break;
            }
        },
        subscribeDataSourceChanged: function(){
            this.variablesSource.subscribePropertyChanged(this._onDataSourcePropertyChanged, this);
        },

        _onDataSourcePropertyChanged: function (event) {
            event.idx = this.itemIdToGraphId[event.property];
            if (event.idx != undefined) {
                this.graphs[event.idx].dataStorage.add(event.newValue.value, event.newValue.sourceTime, event.newValue.statusCode);
            }
            return;
        },

        onDataSourceChanged: function(event){
            if (event.newValue.typeName == "STRING") {
                if ((event.newValue.value) && (event.newValue.value != this.graphs[event.idx].path)) {
                    if (this.graphs[event.idx].path) {
                        this.variablesSource.unsubscribe(this.variablesSource.createVarForStringPen(this.graphs[event.idx].path));
                    }
                    this.variablesSource.addToSubscription(this.variablesSource.createVarForStringPen(event.newValue.value), this.mvc.model.getId());
                    this.itemIdToGraphId['0/' + this.graphs[event.idx].path] = null;
                    this.itemIdToGraphId['0/' + event.newValue.value] = event.idx;
                    this.graphs[event.idx].path = event.newValue.value;
                    this.mvc.model.unsubscribePropertyChanged(this._onDataSourcePropertyChanged);
                    this.variablesSource.unsubscribeAll();
                    this._startupDataExchange();
                    this.graphs[event.idx].dataStorage.clear();
                    this._startDrawing();
                } else {
                    return;
                }
            }
        },
        _onChartBackgroundChanged: function () {
            this.backLayer.grid.canvas.beginRender();
            this.backLayer.grid.draw();
            this.backLayer.grid.canvas.endRender();
        },

        _onShiftChanged: function () {
            this.rightBorder = this.variablesSource.getServerTime();
        },

        onIntervalPagesTurn: function(){
            var interval = this.mvc.model.get(Enums.ParameterRoles.INTERVAL_PAGES_TURN);
            if (interval >= 0) {
                return;
            }
            else {
                this.mvc.model.set(Enums.ParameterRoles.INTERVAL_PAGES_TURN, 100);
                Error.warn(String.format("Интервал листания у тренда {0} принял недопустимое значение {1}. Выставлено значение по умолчанию 100", this.mvc.model.getId(), interval));
            }
        },
        onResampleChanged: function(event){
            var resample = this.mvc.model.get(Enums.ParameterRoles.RESAMPLE_INTERVAL);
            if (this.mvc.model.get(Enums.ParameterRoles.RESAMPLE_INTERVAL) < 0) {
                this.mvc.model.set(Enums.ParameterRoles.RESAMPLE_INTERVAL, 1);
                Error.warn(String.format("Коэффициент прореживания у тренда {0} принял недопустимое значение {1}. Выставлено значение по умолчанию 1", this.mvc.model.getId(), resample));
            }
        },
        onTillChanged: function (value) {
            if (this.initialized && (this.mark > 0) && (value > 0) && (value != this.mark) && (!this.mvc.model.getIsAutoScroll())) {
            //    this.mvc.model.stopAutoscroll();
                    var archiveInfo = this.variablesSource.getArchiveInfo();
                        this.mark = value;
                        return this.getArchiveData(this.mark - this.mvc.model.getInterval(), this.mark);
            }
        },
        _onIsEnabledChanged: function (value) {
            //этот метод не перекрывает оригинальный onIsEnabledChanged
            var isEnabled = this.mvc.model.get(Enums.ParameterRoles.IS_ENABLED);
            this.backLayer.setActive(isEnabled);
            this.linesCanvas.active = isEnabled;
            this.miniLegendCanvas.active = isEnabled;
        },
        _setAutoscrollCursorPosition: function() {
            var shiftInterval = this.mvc.model.get(Enums.ParameterRoles.SHIFT) *
                                this.mvc.model.get(Enums.ParameterRoles.INTERVAL) / 100;
            shiftInterval = shiftInterval != 0 ? shiftInterval : 200;

            this.rightBorder = this.variablesSource.getServerTime() + shiftInterval;
            //move at start
            this._setMark(this.rightBorder);
        },

        onParentActualHeightChanged: function () {
            this.onParentActualWidthChanged();
        },

        onParentActualWidthChanged: function () {
            if (!this.initialized) {
                return;
            }
            if (this.isSizeChanged === false) {
                this.defaultXInterval = this.mvc.model.getInterval();
            }

            var trendLineRange = this.trendEventWrapper.defaultTrendlineRange,
                savedDefaultYRange = this.trendEventWrapper.defaultYRange;

            this._createLayers();
            this._updateYAxis(this.currentYRange);
            this.trendEventWrapper.defaultTrendlineRange = trendLineRange;
            this.trendEventWrapper.defaultInterval = this.defaultXInterval;
            this.trendEventWrapper.defaultYRange = savedDefaultYRange;
            this._positionMinilegendCanvas();
            this.isSizeChanged = true;

            if (this.graphs.length > 0) {
                this._selectTrendline(this.graphs[this.selectedGraphIdx].model);
            }
        },

        onAutoScaleChanged: function (event) {
            if (!this.mvc.model.getIsAutoScale() && this.initialized) {
                var selectedGraphModel = this.graphs[this.selectedGraphIdx].model;
                this._updateYAxis(selectedGraphModel.getStyleSettings());
            }
        },

        onAutoscrollChanged: function () {
            if (this.initialized && this.mark > 0 && this.selectedGraphIdx!=-1) {
                if (this.mvc.model.getIsAutoScroll() === true) {
                    this._setAutoscrollCursorPosition();
                    this._selectTrendline(this.graphs[this.selectedGraphIdx].model);
                    this.graphs[this.selectedGraphIdx].model.isSelected = true;
                    this.isUpdated = false;
                    this.update();
                }
            }
        },
        update: function(){
            this.variablesSource.updateArchiveInfo().then(function () {
                this.getArchiveData(this.mark - this.mvc.model.getInterval(), this.mark).then(function () {
                    this.variablesSource.reatartDataUpdate();
                }.bind(this));
            }.bind(this));
        },
        onIntervalChanged: function () {
            if (this.initialized && this.mark > 0) {
                if (this.mvc.model.previousInterval === 0) {
                    this.mvc.model.previousInterval = this.mvc.model.getInterval();
                }
                if (this.mvc.model.getIsAutoScroll() == true) {
                    this.rightBorder = this.variablesSource.getServerTime();
                }
                if (this.mvc.model.getIsAutoScroll() === false) {
                    this.getArchiveData(this.mark - this.mvc.model.getInterval(), this.mark);
                    this.backLayer.onValueChanged(this.mark);
                }
            }
        },

        onUpdateTimeChanged: function(){
            this.variablesSource.setSourcesOptions({
                interval: this.mvc.model.get(Enums.ParameterRoles.UPDATE_TIME)
            });
            this.variablesSource.startTimer();
        },

        _resetGraphs: function () {
            var interval = this.mvc.model.getInterval();
            this.graphs.forEach(function (serie) {
                serie.reset(interval);
            }, this);
        },

        _getNewMark: function () {
            return this.rightBorder;          
        },

        _setMark: function (mark) {
            this.mark = mark;
            this.mvc.model.set(Enums.ParameterRoles.TILL, mark);
        },

        nextPage: function () {
            this.mvc.model.stopAutoscroll();
            var nextmark = this.mark + this.mvc.model.getInterval() * this.mvc.model.get(Enums.ParameterRoles.INTERVAL_PAGES_TURN) / 100;
                var oldMark = this.mark;
                this._setMark(nextmark);
                return this.getArchiveData(oldMark, nextmark);
        },

        previousPage: function () {
            this.mvc.model.stopAutoscroll();
            var previousMark = this.mark - this.mvc.model.getInterval() * this.mvc.model.get(Enums.ParameterRoles.INTERVAL_PAGES_TURN) / 100;
                //we need to move left point on 2x interval
                var oldMark = this.mark;
                this._setMark(previousMark);
                return this.getArchiveData(oldMark - 2 * this.mvc.model.getInterval(), previousMark);
        },

        firstPage: function () {
            this.mvc.model.stopAutoscroll();
                var archiveInfo = this.variablesSource.getArchiveInfo();
                if (archiveInfo.minStartTime == Number.POSITIVE_INFINITY || archiveInfo.minStartTime == 0) {
                    return when.resolve();
                }
                this._setMark(archiveInfo.minStartTime + this.mvc.model.getInterval());
                return this.getArchiveData(archiveInfo.minStartTime, this.mark);
        },

        lastPage: function () {
            this.mvc.model.stopAutoscroll();
                this._setMark(this.variablesSource.getServerTime());
                return this.getArchiveData(this.variablesSource.getServerTime() - this.mvc.model.getInterval(), this.mark);
        },

        _setEndStartArhive: function () {
            this.mvc.model.set(Enums.ParameterRoles.ARCHIVE_END, this.variablesSource.getServerTime());
            this.mvc.model.set(Enums.ParameterRoles.ARCHIVE_START, this.variablesSource.getArchiveInfo().minStartTime);
        },

        validateProperty: function (propertyName, newProperty) {
            switch (propertyName) {
                case Enums.ParameterRoles.X_MAJOR_TICK_COUNT:
                    this.validator.validateMinMax(
                        {
                            propertyName: propertyName,
                            maxValue: Number.POSITIVE_INFINITY,
                            minValue: 2,
                            property: newProperty
                        });
                    break;
            }
        },
        getArchiveData: function (startTime, endTime) {
            this.verticalCursor.cleanBackLayer();
            return this.variablesSource.getArchiveData(startTime, endTime, this.mvc.model.getInterval() / this.backLayer.getGridDimensions().width * this.mvc.model.get(Enums.ParameterRoles.RESAMPLE_INTERVAL), this.graphs).then(function (data) { 
                if (data && data.length > 0) {
                    this._resetGraphs();
                }
            }.bind(this));
        },

        getServerVarsFromConnectionsSettings: function (connectionSettings) {
            var serverVars = {
                getValues: [],
                setValues: []
            };
            var windowServerVariables = contextResolver.getRootWindow(this).model.serverVariables.getValues;
            connectionSettings.forEach(function (connSetting) {
                if (connSetting.source.type == Enums.connectionType.server && connSetting.target.propertyPath.indexOf(Enums.ParameterRoles.DATA_SOURCE) != -1) {
                    for (var i = 0; i < windowServerVariables.length; i++) {
                        if (windowServerVariables[i].dataSourceId == connSetting.source.dataSource && windowServerVariables[i].itemId == connSetting.source.itemId && windowServerVariables[i].path == connSetting.source.propertyPath) {
                            serverVars.getValues.push($.extend(true, {}, windowServerVariables[i]));
                            break;
                        }
                    }
                }
            }, this);

            return serverVars;
        },

        onDestroy: function () {
            this._super();
            this._stopDrawing();
            this._resetGraphs();
            this.mvc.model.unsubscribePropertyChanged(this._onDataSourcePropertyChanged);
            this.mvc.model.eventTarget.removeAllListeners();
            this.variablesSource.unsubscribeAll();
        }
    });

    return TrendController;
});
