﻿// <copyright file="Alghoritms.js" company="ИнСАТ">
// ИнСАТ, 2014
// </copyright>
// _completedRequest


define(['common/Error'], function (Error) {

    // Берем 1, потому что тут все считается в пикселах
    // Нам больше точность и не нужна
    var EPSILON = 1;
    var previous = 0;

    function Alghoritms() {
    }

    Alghoritms.prototype.euclideanDistance = function (x, y) {
        var x0 = x[0],
            y0 = x[1],
            x1 = y[0],
            y1 = y[1];

        return Math.sqrt(Math.pow(x1 - x0, 2) + Math.pow(y1 - y0, 2));
    }

    Alghoritms.prototype.lineTangent = function (f, s) {
        var x0 = f[0],
            y0 = f[1],
            x1 = s[0],
            y1 = s[1];

        return (y1 - y0) / (x1 - x0);
    }

    Alghoritms.prototype.arePointsClose = function (x, y) {
        return this.euclideanDistance(x, y) <= EPSILON;
    }

    Alghoritms.prototype.areSegmentsIntersect = function (first, second) {
        var a = {};
        a["first"] = {};
        a["first"]["x"] = first[0][0];
        a["first"]["y"] = first[0][1];
        a["second"] = {};
        a["second"]["x"] = first[1][0];
        a["second"]["y"] = first[1][1];


        var b = {};
        b["first"] = {};
        b["first"]["x"] = second[0][0];
        b["first"]["y"] = second[0][1];
        b["second"] = {};
        b["second"]["x"] = second[1][0];
        b["second"]["y"] = second[1][1];
        /* get intersection, if possible */
        if (this._doLinesIntersect(a, b)) {
            var intersection = this._getIntersection(a, b);
            return [intersection["first"]["x"], intersection["first"]["y"]];
        }
        // { "first": { "x": x1, "y": y1 }, "second": { "x": x2, "y": y2 } };
        return [NaN, NaN];
    }

    Alghoritms.prototype._getIntersection = function (a, b) {
        /* the intersection [(x1,y1), (x2, y2)]
           it might be a line or a single point. If it is a line,
           then x1 = x2 and y1 = y2.  */
        var x1, y1, x2, y2;

        if (a["first"]["x"] == a["second"]["x"]) {
            // Case (A)
            // As a is a perfect vertical line, it cannot be represented
            // nicely in a mathematical way. But we directly know that
            //
            x1 = a["first"]["x"];
            x2 = x1;
            if (b["first"]["x"] == b["second"]["x"]) {
                // Case (AA): all x are the same!
                // Normalize
                if (a["first"]["y"] > a["second"]["y"]) {
                    a = { "first": a["second"], "second": a["first"] };
                }
                if (b["first"]["y"] > b["second"]["y"]) {
                    b = { "first": b["second"], "second": b["first"] };
                }
                if (a["first"]["y"] > b["first"]["y"]) {
                    var tmp = a;
                    a = b;
                    b = tmp;
                }

                // Now we know that the y-value of a["first"] is the 
                // lowest of all 4 y values
                // this means, we are either in case (AAA):
                //   a: x--------------x
                //   b:    x---------------x
                // or in case (AAB)
                //   a: x--------------x
                //   b:    x-------x
                // in both cases:
                // get the relavant y intervall
                y1 = b["first"]["y"];
                y2 = Math.min(a["second"]["y"], b["second"]["y"]);
            } else {
                // Case (AB)
                // we can mathematically represent line b as
                //     y = m*x + t <=> t = y - m*x
                // m = (y1-y2)/(x1-x2)
                var m, t;
                m = (b["first"]["y"] - b["second"]["y"]) /
                    (b["first"]["x"] - b["second"]["x"]);
                t = b["first"]["y"] - m * b["first"]["x"];
                y1 = m * x1 + t;
                y2 = y1
            }
        } else if (b["first"]["x"] == b["second"]["x"]) {
            // Case (B)
            // essentially the same as Case (AB), but with
            // a and b switched
            x1 = b["first"]["x"];
            x2 = x1;

            var tmp = a;
            a = b;
            b = tmp;

            var m, t;
            m = (b["first"]["y"] - b["second"]["y"]) /
                (b["first"]["x"] - b["second"]["x"]);
            t = b["first"]["y"] - m * b["first"]["x"];
            y1 = m * x1 + t;
            y2 = y1
        } else {
            // Case (C)
            // Both lines can be represented mathematically
            var ma, mb, ta, tb;
            ma = (a["first"]["y"] - a["second"]["y"]) /
                 (a["first"]["x"] - a["second"]["x"]);
            mb = (b["first"]["y"] - b["second"]["y"]) /
                 (b["first"]["x"] - b["second"]["x"]);
            ta = a["first"]["y"] - ma * a["first"]["x"];
            tb = b["first"]["y"] - mb * b["first"]["x"];
            if (ma == mb) {
                // Case (CA)
                // both lines are in parallel. As we know that they 
                // intersect, the intersection could be a line
                // when we rotated this, it would be the same situation 
                // as in case (AA)

                // Normalize
                if (a["first"]["x"] > a["second"]["x"]) {
                    a = { "first": a["second"], "second": a["first"] };
                }
                if (b["first"]["x"] > b["second"]["x"]) {
                    b = { "first": b["second"], "second": b["first"] };
                }
                if (a["first"]["x"] > b["first"]["x"]) {
                    var tmp = a;
                    a = b;
                    b = tmp;
                }

                // get the relavant x intervall
                x1 = b["first"]["x"];
                x2 = Math.min(a["second"]["x"], b["second"]["x"]);
                y1 = ma * x1 + ta;
                y2 = ma * x2 + ta;
            } else {
                // Case (CB): only a point as intersection:
                // y = ma*x+ta
                // y = mb*x+tb
                // ma*x + ta = mb*x + tb
                // (ma-mb)*x = tb - ta
                // x = (tb - ta)/(ma-mb)
                x1 = (tb - ta) / (ma - mb);
                y1 = ma * x1 + ta;
                x2 = x1;
                y2 = y1;
            }
        }

        return { "first": { "x": x1, "y": y1 }, "second": { "x": x2, "y": y2 } };
    }

    Alghoritms.prototype._doLinesIntersect = function (a, b) {
        var box1 = this._getBoundingBox(a);
        var box2 = this._getBoundingBox(b);
        return this._doBoundingBoxesIntersect(box1, box2)
                && this._lineSegmentTouchesOrCrossesLine(a, b)
                && this._lineSegmentTouchesOrCrossesLine(b, a);
    }

    Alghoritms.prototype._isPointOnLine = function (a, b) {
        // Move the image, so that a.first is on (0|0)
        var aTmp = { "first": { "x": 0, "y": 0 }, "second": { "x": a.second.x - a.first.x, "y": a.second.y - a.first.y } };
        var bTmp = { "x": b.x - a.first.x, "y": b.y - a.first.y };
        var r = this._crossProduct(aTmp.second, bTmp);
        return Math.abs(r) < EPSILON;
    }

    Alghoritms.prototype._crossProduct = function (a, b) {
        return a.x * b.y - b.x * a.y;
    }

    Alghoritms.prototype._isPointRightOfLine = function (a, b) {
        // Move the image, so that a.first is on (0|0)
        var aTmp = { "first": { "x": 0, "y": 0 }, "second": { "x": a.second.x - a.first.x, "y": a.second.y - a.first.y } };
        var bTmp = { "x": b.x - a.first.x, "y": b.y - a.first.y };
        return this._crossProduct(aTmp.second, bTmp) < 0;
    }


    Alghoritms.prototype._doBoundingBoxesIntersect = function (a, b) {
        return a[0].x <= b[1].x && a[1].x >= b[0].x && a[0].y <= b[1].y
                && a[1].y >= b[0].y;
    }

    Alghoritms.prototype._getBoundingBox = function (a) {
        return [{
            "x": Math.min(a["first"]["x"], a["second"]["x"]),
            "y": Math.min(a["first"]["y"], a["second"]["y"])
        },
                {
                    "x": Math.max(a["first"]["x"], a["second"]["x"]),
                    "y": Math.max(a["first"]["y"], a["second"]["y"])
                }];
    }

    Alghoritms.prototype._lineSegmentTouchesOrCrossesLine = function (a, b) {
        return this._isPointOnLine(a, b.first)
                || this._isPointOnLine(a, b.second)
                || (this._isPointRightOfLine(a, b.first) ^ this._isPointRightOfLine(a,
                        b.second));
    }

    Alghoritms.prototype.getIntersection = function (flfp, flsp, slfp, slsp) {
        return this.areSegmentsIntersect([[flfp.x, flfp.y], [flsp.x, flsp.y]],
            [[slfp.x, slfp.y], [slsp.x, slsp.y]]);
    }   

    /** 
		* Checks if two segments are intersect
		* @function areSegmentsIntersect	
		* @this {Alghoritms}
        * @param first{array} Array of first segment's points
        * @param second{array} Array of second segment's points
        * @returns Coordinates of intersection point, or [NaN, NaN], if segments aren't intersect.
	*/
    Alghoritms.prototype.areSegmentsIntersect2 = function (first, second) {
        var fl = first.length,
            sl = second.length;

        if (fl < 2 || sl < 2) {
            Error.onerror('Minimum two points needed for checking intersection');
        }

        var ffp = first[0],
            fsp = first[1],
            sfp = second[0],
            ssp = second[1],
            a = fsp[0] - ffp[0],
            b = fsp[1] - ffp[1],
            c = ssp[0] - sfp[0],
            d = ssp[1] - sfp[1],
            e = ffp[0] * b - ffp[1] * a,
            f = sfp[0] * d - sfp[1] * c;

        if (b === 0 && d === 0) {
            return [NaN, NaN];
        } else if (b === 0) {
            return this.areSegmentsIntersect(second, first);
        }
        

        // if we have two segments with equal X coordinates
        // we need only to check y coord
        // result point will look like [x1 + (x2 - x1) / 2, y1 + (y2 - y1) / 2]
        if (ffp[0] === sfp[0] && fsp[0] === ssp[0]) {
            if ((ffp[1] - sfp[1]) * (fsp[1] - ssp[1]) < 0) {
                return [ffp[0] + (fsp[0] - ffp[0]) / 2, ffp[1] + (fsp[1] - ffp[1]) / 2];
            }
        }

        if ((fsp[0] - ssp[0]) * (ffp[1] - sfp[1]) - (ffp[0] - sfp[0]) * (fsp[1] - ssp[1]) === 0) {
            return [NaN, NaN];
        }

        var y = (b * f - d * e) / (d * a - b * c),
            ipf = (e + (a * y)) / b,
            // инвертируем ось Y потому что на экране другая система координат,
            // а СЛАУ решалась в несмещенной декартовой
            ips = -y;

        if ((first[0][0] <= ipf && ipf <= first[1][0]) && (second[0][0] <= ipf && second[1][0] >= ipf)) {
           // if ((first[0][1] <= y && y <= first[1][1]) && (second[0][1] <= y && second[1][1] >= y)) {
                return [ipf, ips];
            //}            
        }

        return [NaN, NaN];
    };

    /** 
       * Checks if point and line are intersect
       * @function arePointAndLineIntersect
       * @this {Alghoritms}
       * @param line{array} Array of two line points;
       * @param point{object} Object with x and y point coordinates
       * @returns Coordinates of intersection point
   */
    Alghoritms.prototype.getPerpendicularIntersectPoint = function (line, point) {
        var x1 = line[0].x,
            y1 = line[0].y,
            x2 = line[1].x,
            y2 = line[1].y,
            x3 = point.x,
            y3 = point.y;

        // first convert line to normalized unit vector
        var dx = x2 - x1;
        var dy = y2 - y1;
        var mag = Math.sqrt(dx * dx + dy * dy);
        dx /= mag;
        dy /= mag;

        // translate the point and get the dot product
        var lambda = (dx * (x3 - x1)) + (dy * (y3 - y1));
        x4 = (dx * lambda) + x1;
        y4 = (dy * lambda) + y1;

        return { x: x4, y: y4 };
    }

    /** 
        * Checks if point belongs to rectangle region
        * @function isPointInRegion	
        * @this {Alghoritms}
        * @param point{array} Array of first segment's points
        * @param region{array} Array of second segment's points
        * @returns Coordinates of intersection point, or undefined, if segments aren't intersect.
    */
    Alghoritms.prototype.isPointInRegion = function (point, region) {
        if (point.x > region.x1 && point.x < region.x2
                     && point.y > region.y1 && point.y < region.y2) {
            return true;
        }

        return false;
    }


    /** 
* Binary search in array of objects
* @function binarySearch
* @this {Alghoritms}
* @param arr{array} array of objects
* @param keyField{string} object key name
* @param value{string} object key value to compare with
* @returns index of value in array, or -1 if not found
*/

    Alghoritms.prototype.binarySearch = function (arr, keyField, value) {
        var lowBound = 0,
            highBound = arr.length - 1,
            mid = 0;
        if (value == null) {
            Error.onerror("Value is not specified for binary search.");
            return { lowBound: highBound, highBound: lowBound };
        }

        if (arr.length === 1) {
            return { lowBound: mid, highBound: mid };
        }

        while (lowBound <= highBound) {
            if (!arr[mid].hasOwnProperty(keyField)) {
                Error.onerror("No such property in array objects(binary search)");
                return { lowBound: highBound, highBound: lowBound };
            }

            mid = Math.round((lowBound + highBound) / 2);
            //the element we search is located to the right from the mid point
            if (arr[mid][keyField] < value) {
                lowBound = mid + 1;
                continue;
            //the element we search is located to the left from the mid point
            } else if (arr[mid][keyField] > value) {
                highBound = mid - 1;
                continue;
            }
            //at this point low and high bound are equal and we have found the element or
            //arr[mid] is just equal to the value => we have found the searched element
            else {
                return { lowBound: mid, highBound: mid };
            }
        }

        return { lowBound: highBound, highBound: lowBound };//value not found
    }

    /** 
        * Returns an Y coordinate of point by given X
        * @function getYOnLine	
        * @this {Alghoritms}
        * @param line{array} Two-dimensonal array of line points
        * @param x{array} X coordinate of point
        * @returns Y coordinate of point.
    */
    Alghoritms.prototype.getYOnLine = function (line, x) {
        var x0 = line[0][0],
            x1 = line[1][0],
            y0 = line[0][1],
            y1 = line[1][1],
            a = x1 - x0,
            b = x - x0,
            c = y1 - y0;
        
        // случай с вертикальной прямой
        // бесконечное множество точек (если х лежит на прямой)
        // или прямая не проходит через точку
        if (a === 0) {
            return NaN;
        // горизонтальная прямая, подойдет любой y
        } else if (c === 0) {
            return y1;
        }

        return ((b * c) / a + y0);
    }

    Alghoritms.prototype.isPointOnLine = function (line, point) {
        var x0 = line[0][0],
            x1 = line[1][0],
            y0 = line[0][1],
            y1 = line[1][1],
            x = point[0],
            y = point[1];

        return this._isPointOnLine({ first: { x: x0, y: y0 }, second: { x: x1, y: y1 } }, { x: x, y: y });
    }

    Alghoritms.prototype.getScreenX = function (mark, interval, lastMark, width) {
        var x = 1 - ((lastMark - mark) / interval);
        if (isNaN(x)) {
            x = 0;
        }

        return x * width;
    }

    Alghoritms.prototype.getScreenY = function (value, pxHeight, range) {
        //convert value to canvas y
        return pxHeight * (1 - ((value - range[0]) / (range[1] - range[0])));
    }

    Alghoritms.prototype.getPointScreenCoords = function (item, interval, mark, canvasWidth, height, range) {
        return {
            x: this.getScreenX(item.timepoint, interval, mark, canvasWidth),
            y: this.getScreenY(item.value, height, range),
            sc: item.quality,
            timepoint: item.timepoint,
            value: value
        }
    }

    Alghoritms.prototype.getY = function (screenY, pxHeight, range) {
        //convert canvas y to value
        //range[0] - yAxis min, range[1] - yAxis max
        //т.к. ось y рисуется сверху вниз, то полученный range отнимаем от максимума
        return range[1] - ((range[1] - range[0]) * (screenY / pxHeight));
    }

    Alghoritms.prototype.getX = function (screenX, pxWidth, beginRange, endRange) {
        return (beginRange + (beginRange - endRange) * (screenX / pxWidth));
    }

    return new Alghoritms();
});