// <copyright file="Canvas.js" company="ИнСАТ">
// ИнСАТ, 2014
// </copyright>
// 

define(['when', 'common/Error', 'common/Enums', 'libs/hammer'],
    function (when, Error, Enums, hammer) {

        // http://paulirish.com/2011/requestanimationframe-for-smart-animating/
        // http://my.opera.com/emoller/blog/2011/12/20/requestanimationframe-for-smart-er-animating

        // requestAnimationFrame polyfill by Erik Möller. fixes from Paul Irish and Tino Zijdel

        // MIT license

        (function () {
            var lastTime = 0;
            var vendors = ['ms', 'moz', 'webkit', 'o'];
            for (var x = 0; x < vendors.length && !window.requestAnimationFrame; ++x) {
                window.requestAnimationFrame = window[vendors[x] + 'RequestAnimationFrame'];
                window.cancelAnimationFrame = window[vendors[x] + 'CancelAnimationFrame']
                || window[vendors[x] + 'CancelRequestAnimationFrame'];
            }
            if (!window.requestAnimationFrame)
                window.requestAnimationFrame = function (callback, element) {
                    var currTime = new Date().getTime();
                    var timeToCall = Math.max(0, 16 - (currTime - lastTime));
                    var id = window.setTimeout(function () { callback(currTime + timeToCall); },
                    timeToCall);
                    lastTime = currTime + timeToCall;
                    return id;
                };
            if (!window.cancelAnimationFrame)
                window.cancelAnimationFrame = function (id) {
                    clearTimeout(id);
                };
        }());


        var requestAnimationFrame = window.requestAnimationFrame;

        function Canvas(id, dispatchFn) {
            this.id = id;
            this.dispatchFn = dispatchFn;
            this.renderQueue = [];
            this.canvas = this._createCanvas();
            this.eventListeners = {};
            this.requestAnimationFrameQueueMaxLength = 10;
            this.requestAnimationFrameQueue = [];
            this.renderLoopBind = this.renderLoop.bind(this);
            this.disposed = false;
            this.active = true;
            this.renderLoop();
        };

        Canvas.prototype.drawCircle = function (options) { //xCenter, yCenter, radius, 0, 2*Math.PI, strokeStyle, fillStyle
            if (options.strokeStyle) {
                this.drawStrokeCircle(options);
            }
            if (options.fillStyle) {
                this.drawFillCircle(options);
            }
            return this;
        };

        Canvas.prototype.beginRender = function () {
            if (this.renderQueue.length > 0) {
                Error.onerror("Canvas " + this.id + " : render queue is not empty.");
                this.renderQueue = [];
            }
        };

        Canvas.prototype.drawRect = function (options) {
            if (options.strokeStyle) {
                this.drawStrokeRect(options);
            }
            if (options.fillStyle) {
                this.drawFillRect(options);
            }
            // for chaining
            return this;
        }

        Canvas.prototype.drawLine = function (options) {
            if (options.lineStyle != "none") {
                this._drawLineInternal(options);
            }
            // for chaining           
            return this;
        }

        Canvas.prototype._drawLineInternal = function (options) {

            this.renderQueue.push(function (x1, y1, x2, y2, strokeStyle, lineWidth, lineStyle) {
                return function () {
                    //this will point to canvas 2d context 
                    var prevStyle, prevWidth,
                        ls = lineStyle || Enums.LineStyle.solid,
                        llength = Math.sqrt(Math.pow(x2 - x1, 2) + Math.pow(y2 - y1, 2));

                    if (this.strokeStyle !== strokeStyle) {
                        prevStyle = this.strokeStyle;
                        this.strokeStyle = strokeStyle;
                    }
                    if (this.lineWidth !== lineWidth) {
                        prevWidth = this.lineWidth;
                        this.lineWidth = lineWidth;
                    }

                    if (ls === Enums.LineStyle.dashed) {
                        var dashSize = Math.min(1, lineWidth) * 4,
                            gapSize =  dashSize;
                        this.setLineDash([dashSize, gapSize]);
                    } else if (ls === Enums.LineStyle.dotted) {
                        var dashSize = Math.min(1, lineWidth), gapSize = dashSize * lineWidth * 5;
                        this.setLineDash([dashSize, gapSize]);
                        this.lineCap = 'round';
                    }

                    this.beginPath();
                    this.moveTo(x1, y1);
                    this.lineTo(x2, y2);
                    this.stroke();
                    this.closePath();

                    if (this.strokeStyle !== strokeStyle) {
                        this.strokeStyle = prevStyle;
                    }
                    if (this.lineWidth !== lineWidth) {
                        this.lineWidth = prevWidth;
                    }
                    if (ls === Enums.LineStyle.dashed) {
                        // reset line dash
                        this.setLineDash([0, 0]);
                    } else if (ls === Enums.LineStyle.dotted) {
                        this.setLineDash([0, 0]);
                        this.lineCap = 'butt';
                    }
                }
            }(options.x1, options.y1, options.x2, options.y2, options.strokeStyle, options.lineWidth, options.lineStyle));
        }

        Canvas.prototype.drawStrokeCircle = function (options) {
            this.renderQueue.push(function (xCenter, yCenter, radius, strokeStyle) {
                return function () {
                    var prevStyle = this.strokeStyle
                    //this will point to canvas 2d context
                    this.beginPath();
                    this.arc(xCenter, yCenter, radius, 0, 2 * Math.PI);
                    this.strokeStyle = strokeStyle;
                    this.stroke();
                    this.strokeStyle = prevStyle;
                }
            }(options.xCenter, options.yCenter, options.radius, options.strokeStyle));
            // for chaining
            return this;
        };

        Canvas.prototype.drawFillCircle = function (options) {
            this.renderQueue.push(function (xCenter, yCenter, radius, fillStyle) {
                return function () {
                    var prevFill = this.fillStyle;
                    //this will point to canvas 2d context              
                    this.beginPath();
                    this.arc(xCenter, yCenter, radius, 0, 2 * Math.PI);
                    this.fillStyle = fillStyle;
                    this.fill();
                    this.fillStyle = prevFill;
                }
            }(options.xCenter, options.yCenter, options.radius, options.fillStyle));
            // for chaining
            return this;
        };

        Canvas.prototype.drawStrokeRect = function (options) {
            this.renderQueue.push(function (x, y, width, height, strokeStyle) {
                return function () {
                    var prevStyle = this.strokeStyle;
                    this.strokeStyle = strokeStyle;
                    //this will point to canvas 2d context
                    this.strokeRect(x, y, width, height);
                    this.strokeStyle = prevStyle;
                }
            }(options.x, options.y, options.width, options.height, options.strokeStyle));
            // for chaining
            return this;
        };

        Canvas.prototype.beginPath = function (options) {
            this.renderQueue.push(function () {
                //this will point to canvas 2d context              
                this.beginPath();
            });

            // for chaining
            return this;
        };

        Canvas.prototype.closePath = function (options) {
            this.renderQueue.push(function () {
                //this will point to canvas 2d context              
                this.closePath();
            });
            // for chaining
            return this;
        };

        Canvas.prototype.fill = function (options) {
            this.renderQueue.push(function () {
                //this will point to canvas 2d context              
                this.fill();
            });

            // for chaining
            return this;
        };

        Canvas.prototype.stroke = function (options) {
            this.renderQueue.push(function () {
                //this will point to canvas 2d context              
                this.stroke();
            });
            // for chaining
            return this;
        };

        Canvas.prototype.drawFillRect = function (options) {
            this.renderQueue.push(function (x, y, width, height, fill) {
                return function () {
                    var prevFill = this.fillStyle;
                    this.fillStyle = fill;
                    //this will point to canvas 2d context              
                    this.fillRect(x, y, width, height);
                    this.fillStyle = prevFill;
                }
            }(options.x, options.y, options.width, options.height, options.fillStyle));
            // for chaining
            return this;
        };

        Canvas.prototype.clearRect = function (options) {
            this.renderQueue.push(function (x, y, width, height) {
                return function () {
                    //this will point to canvas 2d context              
                    this.clearRect(x, y, width, height);
                }
            }(options.x, options.y, options.width, options.height));
            // for chaining
            return this;
        };

        Canvas.prototype.setStrokeStyle = function (strokeStyle) {
            this.renderQueue.push(function (strokeStyle) {
                return function () {
                    //this will point to canvas 2d context
                    this.strokeStyle = strokeStyle;
                }
            }(strokeStyle));
            // for chaining
            return this;
        }

        Canvas.prototype.setLineWidth = function (lineWidth) {
            this.renderQueue.push(function (lineWidth) {
                return function () {
                    //this will point to canvas 2d context
                    this.lineWidth = lineWidth;
                }
            }(lineWidth));
            // for chaining
            return this;
        }

        Canvas.prototype.moveTo = function (options) {
            this.renderQueue.push(function (x, y) {
                return function () {
                    this.moveTo(x, y)
                }
            }(options.x, options.y));
            // for chaining
            return this;
        }

        Canvas.prototype.lineTo = function (options) {
            this.renderQueue.push(function (x, y) {
                return function () {
                    this.lineTo(x, y)
                }
            }(options.x, options.y));
            // for chaining
            return this;
        }

        Canvas.prototype.drawImage = function (options) {
            if (!(options.img instanceof Canvas)) {
                Error.onerror("options.img must be an instance of Canvas class");
            }

            this.renderQueue.push(function (img, x, y, width, height,
                sx, sy, swidth, sheight) {

                return function () {
                    if (width === undefined || height === undefined) {
                        this.drawImage(img.canvas, x, y);
                    } else if (width !== undefined && height !== undefined) {
                        this.drawImage(img.canvas, x, y, width, height);
                    } else {
                        this.drawImage(img.canvas, sx, sy, swidth, sheight,
                            x, y, width, height);
                    }
                }
            }(options.img, options.x, options.y, options.width, options.height,
                options.sx, options.sy, options.swidth, options.sheight));
            // for chaining
            return this;
        }

        Canvas.prototype.setFillStyle = function (fillStyle) {
            this.renderQueue.push(function (fillStyle) {
                return function () {
                    //this will point to canvas 2d context
                    this.fillStyle = fillStyle;
                }
            }(fillStyle));
            // for chaining
            return this;
        }

        Canvas.prototype.endRender = function () {
            if (this.requestAnimationFrameQueue.length > this.requestAnimationFrameQueueMaxLength) {
                this.requestAnimationFrameQueue.splice(0, 1);
            }
            this.requestAnimationFrameQueue.push(this.renderQueue);
            this.renderQueue = [];
        }

        Canvas.prototype.renderInternal = function (renderQueue, canvasContext) {
            canvasContext.mozImageSmoothingEnabled = false;
            canvasContext.webkitImageSmoothingEnabled = false;
            for (var i = 0, l = renderQueue.length; i < l; i++) {
                renderQueue[i].call(canvasContext);
            }
        }

        Canvas.prototype.getNode = function () {
            return this.canvas;
        }

        Canvas.prototype.measureText = function (text) {
            return parseFloat(this._getContext().measureText(text).width);
        }

        Canvas.prototype.getFontSize = function () {
            return parseFloat(this._getContext().font);
        }

        Canvas.prototype.setFont = function (font) {
            this._getContext().textBaseline = "top";
            this._getContext().font = font;
            return this;
        }

        Canvas.prototype.setFontStyle = function (font) {
            this.renderQueue.push(function (fillStyle) {
                return function () {
                    this.font = font;
                }
            }(font));
            // for chaining
            return this;
        }

        Canvas.prototype.setSize = function (size) {
            this.canvas.width = size.width;
            this.canvas.height = size.height;
            return this;
        }

        Canvas.prototype.getSize = function () {
            return {
                width: this.canvas.width,
                height: this.canvas.height
            }
        }

        Canvas.prototype._getContext = function () {
            return this.canvas.getContext('2d');
        }

        Canvas.prototype.setPosition = function (options) {
            this.canvas.style.left = (options.x || 0) + 'px';
            this.canvas.style.top = (options.y || 0) + 'px';
            return this;
        }

        Canvas.prototype.fillText = function (options) {
            this.renderQueue.push(function (text, x, y, maxWidth, angle) {
                return function () {
                    if (options.angle !== undefined && options.angle !== 0) {
                        this.save();
                        this.translate(x, y);
                        this.rotate(angle * Math.PI / 180);
                        this.textAlign = "center";
                        this.fillText(text, 0, 0, maxWidth || parseFloat(this.measureText(text).width));
                        this.restore();
                    } else {
                        // добавляем maxWidth для WebKit, иначе ничего не нарисуется                 
                        this.fillText(text, x, y, maxWidth || parseFloat(this.measureText(text).width));
                    }

                }
            }(options.text, options.x, options.y, options.maxWidth, options.angle));

            return this;
        }

        Canvas.prototype.getRealWidth = function () {
            return this.canvas.width;
        }

        Canvas.prototype.setNodeSize = function (size) {
            this.canvas.style.width = (size.width || 0);
            this.canvas.style.height = (size.height || 0);
            return this;
        }

        Canvas.prototype.clear = function () {
            var size = this.getSize();
            this.clearRect({
                x: 0,
                y: 0,
                width: size.width,
                height: size.height
            });
        }

        Canvas.prototype.addHitRegion = function (eventType, callback) {
            var listeners = this._getEventListeners(eventType);
            listeners.push(callback);
        }

        Canvas.prototype._getEventListeners = function (eventType) {
            if (this.eventListeners[eventType] === undefined) {
                this.eventListeners[eventType] = [];
            }

            return this.eventListeners[eventType];
        }

        Canvas.prototype._processEvent = function (eventType, event) {
            var listeners = this._getEventListeners(eventType);
            if (this.active === false) {
                return;
            }

            var e = {};
            if (event.srcEvent) {
                event = event.srcEvent;
            } else if (event instanceof $.Event) {
                e.clientX = event.clientX;
                e.clientY = event.clientY;
            }

            //в FF на desktop не определен TouchEvent
            //https://bugzilla.mozilla.org/show_bug.cgi?id=888304
            if (window.TouchEvent && event instanceof TouchEvent) {
                e.clientX = event.changedTouches[0].clientX;
                e.clientY = event.changedTouches[0].clientY;
            } else if (event instanceof MouseEvent) {
                e.clientX = event.clientX;
                e.clientY = event.clientY;
            }

            var processed = false;
            var offset = $(this.canvas).offset();
            var relativeCoord = {
                x: e.clientX - offset.left,
                y: e.clientY - offset.top
            };

            var i;
            for (i = 0; i < listeners.length; i++) {
                processed = listeners[i](relativeCoord, eventType, event);
                if (processed === true) {
                    return true;
                }

            }

            this.dispatchFn(this.id, eventType, event);

            return processed;
        }

        Canvas.prototype._createCanvas = function () {
            var self = this;
            var canvas = document.createElement('canvas');
            canvas.style.position = 'absolute';
            this._bindEvents(self, canvas);

            return canvas;
        }

        Canvas.prototype._bindEvents = function (self, canvas) {
            var mc = hammer(canvas, {
                dragLockToAxis: true,
                dragBlockHorizontal: true
            });
            $(canvas).on('contextmenu', function (event) { event.preventDefault(); });
            $(canvas).on('mousedown', function (event) { self._processEvent(Enums.EventType.mouseDown, event); });
            $(canvas).on('mouseup', function (event) { self._processEvent(Enums.EventType.mouseUp, event); });
            $(canvas).on('mousemove', function (event) { self._processEvent(Enums.EventType.mouseMove, event); });
            mc.get('pinch').set({ enable: true });
            mc.on('tap', function (event) { self._processEvent(Enums.EventType.click, event); });
            mc.on('swipeleft', function (event) { self._processEvent(Enums.SwipeDirection.swipeLeft, event); });
            mc.on('swiperight', function (event) { self._processEvent(Enums.SwipeDirection.swipeRight, event); });
            mc.on('panstart', function (event) { self._processEvent(Enums.PanType.panStart, event); });
            mc.on('panend', function (event) {
                self._processEvent(Enums.PanType.panEnd, event);
                event.preventDefault();
            });
            mc.on('pinchmove', function (event) {
                self._processEvent(Enums.PinchDirectional.pinchMove, event);
                event.preventDefault();
            });
            mc.on('pinchin', function (event) {
                self._processEvent(Enums.PinchDirectional.pinchIn, event);
                event.preventDefault();
            });
            mc.on('pinchstart', function (event) {
                self._processEvent(Enums.PinchDirectional.pinchStart, event);
                event.preventDefault();
            });
            mc.on('pinchout', function (event) {
                self._processEvent(Enums.PinchDirectional.pinchOut, event);
                event.preventDefault();
            });
            mc.on('pinchend', function (event) {
                self._processEvent(Enums.PinchDirectional.pinchEnd, event);
                event.preventDefault();
            });


        }

        Canvas.prototype.dispose = function () {
            this.disposed = true;
            if (this.canvas !== undefined && this.canvas.parentNode !== null) {
                this.canvas.parentNode.removeChild(this.canvas);
            }

            this.canvas = null;
            this.removeAllListeners();
            this.requestAnimationFrameQueue = [];
        }

        Canvas.prototype.removeAllListeners = function () {
            this.eventListeners = {};
        }

        Canvas.prototype.renderLoop = function () {
            var i;

            if (this.disposed) {
                return;
            }

            for (i = 0; i < this.requestAnimationFrameQueue.length; i++) {
                this.renderInternal(this.requestAnimationFrameQueue[i], this.canvas.getContext('2d'));
            }

            this.requestAnimationFrameQueue = [];

            requestAnimationFrame(this.renderLoopBind, this.canvas);
        }

        return Canvas;
    });