// <copyright file="TrendlineBase.js" company="ИнСАТ">
// ИнСАТ, 2014
// </copyright>


define(['base/ObservableObject', 'common/Enums', 'common/Appearence',
    'controls/Trend/modules/models/TrendlineModel', 'controls/Trend/modules/Alghoritms', 'controls/Trend/modules/DisplaceQueue'],

    function (ObservableObject, Enums, Appearence, TrendlineModel, Alghoritms, DisplaceQueue) {
        var suffix = 'Changed';

        var TrendlineBase = ObservableObject.extend({
            lineColorOnDisabledTrend: 'black', //IE 10+ do not support grayscale
            init: function (idx, parent, model) {             
                this.parent = parent;
                this._super();
                this._pointsCache = [];
                this.dataStorage = new DisplaceQueue();
                this._idx = idx;
                this.model.subscribePropertyChanged(this.modelPropertyChanged, this);
            },

            modelPropertyChanged: function (event) {
                if (this['on' + event.property + suffix] !== undefined) {
                    this['on' + event.property + suffix](event);
                }
            },

            setCanvas: function (canvas) {
                canvas.addHitRegion(Enums.EventType.click, this._processLineClick.bind(this));
            },

            onIsVisibleChanged: function (event) {
            },

            update: function (canvas, canvasWidth, range, interval, mark, currTimestamp) {
                this.updateLine(canvas, canvasWidth, range, interval, mark, currTimestamp);
            },

            updateLine: function (canvas, canvasWidth, range, interval, mark, currTimestamp) {

                this._pointsCache = [];

                if (!this.model.getIsVisible()) {
                    return;
                }

                this.dataStorage.setInterval(interval);
                this.dataStorage.cleanOldData(mark);
                var data = this.dataStorage.getAll();
                this.canvasWidth = canvasWidth;

                var height = canvas.getSize().height,
                    prevPoint = undefined;

                for (var i = 0, l = data.length; i < l; ++i) {
                    var currentPoint = Alghoritms.getPointScreenCoords(data[i], interval, mark, canvasWidth, height, range);
                    if (currentPoint.x > canvasWidth)
                        break;
                    if (prevPoint != undefined) {
                        var points = this._getPointsByJoinType(prevPoint, currentPoint);
                        this._pointsCache = this._pointsCache.concat(points);
                        this._draw(canvas, prevPoint, points);
                    } else {
                        this._pointsCache.push(currentPoint);
                    }

                    prevPoint = currentPoint;
                }

                if (this._pointsCache.length > 0 &&
                    this._pointsCache[this._pointsCache.length - 1].x < 0 &&
                    this.model.drawIfNoData()) {
                    //на видимом интервале пусто
                    var fakeStartPoint = { x: 0, y: this._pointsCache[this._pointsCache.length - 1].y, sc: this._pointsCache[this._pointsCache.length - 1].sc };
                    this._draw(canvas, this._pointsCache[this._pointsCache.length - 1], [fakeStartPoint]);
                    this._pointsCache.push(fakeStartPoint);
                }

                if (this._pointsCache.length > 0 &&
                    this._pointsCache[this._pointsCache.length - 1].x < canvasWidth &&
                    this.model.drawIfNoData()) {
                    //если не дотягивает до конца интервала
                    //но учитываем, что мы не можем дорисовывать в будущее 
                    //т.е если правая граница интервала в будущем - то можем дорисовать только 
                    //до настоящего времени
                    var lastPoint = this._pointsCache[this._pointsCache.length - 1],
                        rightX;
                    if (mark <= currTimestamp) {
                        rightX = canvasWidth;
                    } else {
                        rightX = Alghoritms.getPointScreenCoords({
                            timepoint: currTimestamp,
                            value: lastPoint.value,
                            quality: lastPoint.sc
                        }, interval, mark, canvasWidth, height, range).x;
                    }

                    var fakeEndPoint = { x: rightX, y: lastPoint.y, sc: lastPoint.sc };
                    this._draw(canvas, lastPoint, [fakeEndPoint]);
                    this._pointsCache.push(fakeEndPoint);

                    data = null;
                }
            },
           
            _draw: function (canvas, startPoint, points) {
                var prevPoint = startPoint;
                for (var i = 0, l = points.length; i < l; i++) {
                    this._drawLine(canvas, prevPoint, points[i]);
                    prevPoint = points[i];
                }
            },

            _drawLine: function (canvas, fromPoint, toPoint) {
                var lineWidth = this.model.getThickness(),
                   ls, selColor, strokeStyle;

                if (!this.parent.mvc.model.getIsEnabled()) {
                    selColor = strokeStyle = this.lineColorOnDisabledTrend;
                } else {
                    selColor = this.model.getSelectionColor();
                    strokeStyle = Appearence.color.toCssColor(this.model.getLineColor());
                }
               
                if (fromPoint.sc === Enums.DataQuality.good || (!fromPoint.sc)) { //INSAT: значения, у которых нет отметки качества, будем считать по умолчанию хорошими
                    ls = this.model.get(Enums.ParameterRoles.LINE_STYLE);
                    // по просьбе Виктора, все что не 0 - плохое.
                } else {
                    ls = this.model.getFalseyValuesLinestyle();

                    if (ls === Enums.BorderStyleType.None) {
                        return;
                    }
                }

                ls = Appearence.border.getBorderStyle(ls);

                canvas.drawLine({
                    strokeStyle: this.model.isSelected ? selColor : strokeStyle,
                    x1: fromPoint.x > 0 ? fromPoint.x : 0,
                    y1: fromPoint.y,
                    x2: toPoint.x,
                    y2: toPoint.y,
                    lineWidth: lineWidth,
                    lineStyle: ls
                });
                
            },

            select: function () {
                this.eventTarget.fire({
                        type: Enums.TrendEvents.selectTrendline,                      
                        idx: this._idx
                    });
            },

            _processLineClick: function (coord) {

                if (!this.model.getIsVisible()) {
                    return;
                }

                //если на канвасе нету линий (н-р залезли в будущее
                //или остановили автопрокрутку, а у атрибута
                //выключено архиварование
                if (this._pointsCache[0] === undefined) {
                    return false;
                }

                if (this._pointsCache[0].x > coord.x || this._pointsCache[this._pointsCache.length - 1].x < coord.x) {
                    // точка совсем не на доступном сейчас отрезке
                    // Vlad: это значит, что мы клацнули по канвасу, но попали перед линией
                    return false;
                }

                var lineDiap = Alghoritms.binarySearch(this._pointsCache, 'x', coord.x);
                if (lineDiap.lowBound === lineDiap.highBound) {
                    //т.е попали прямо в точку
                    return true;
                }

                var T = this.model.get(Enums.ParameterRoles.LINE_WIDTH);;

                //берем ближайшие две точки
                var firstPoint = this._pointsCache[lineDiap.lowBound];
                var secondPoint = this._pointsCache[lineDiap.highBound];

                if (!Alghoritms.isPointInRegion(coord, {
                    x1: firstPoint.x,
                    y1: firstPoint.y - T / 2,
                    x2: secondPoint.x,
                    y2: secondPoint.y + T / 2
                })) {
                    return false;
                }

                var vectAB = [secondPoint.x - firstPoint.x, secondPoint.y - firstPoint.y],
                    vectAC = [coord.x - firstPoint.x, coord.y - firstPoint.y],
                    scalarACAB = vectAC[0] * vectAB[0] + vectAC[1] * vectAB[1],
                    sqrnormAB = Math.pow(Math.sqrt(vectAB[0] * vectAB[0] + vectAB[1] * vectAB[1]), 2),
                    sqrnormAC = Math.pow(Math.sqrt(vectAC[0] * vectAC[0] + vectAC[1] * vectAC[1]), 2),
                    sqrT = T * T;

                if (scalarACAB > 0 && scalarACAB < sqrnormAB) {
                    if ((sqrnormAB * sqrnormAC) <= (sqrT * sqrnormAB + scalarACAB * scalarACAB)) {                        
                        this.select();
                        return true;
                    }
                }

                return false;
            },

            _getPointsByJoinType: function (prevPoint, currentPoint) {
                var lineJoin = this.model.getLineJoin();
                switch (lineJoin) {
                    case Enums.LineJoin.Line:
                        return [currentPoint];
                    case Enums.LineJoin.StairsUp:
                        //from previous without previous itself
                        return [{ x: currentPoint.x, y: prevPoint.y, sc: currentPoint.sc }, currentPoint];
                    case Enums.LineJoin.StairsDown:
                        //from previous without previous itself
                        return [{ x: prevPoint.x, y: currentPoint.y, sc: currentPoint.sc }, currentPoint];
                    default:
                        return [currentPoint];
                }
            },

            reset: function(interval){
                this.dataStorage.clear();
                this.dataStorage.setInterval(interval);
                this._pointsCache = [];
            },

            setInterval: function (interval) {
                this.dataStorage.setInterval(interval);
            },

            hasTimepoints: function () {
                return this.dataStorage.getAll().length > 0;
            },

            getMin: function () {
                return this.dataStorage.getMin().value;
            },
            getMax: function () {
                return this.dataStorage.getMax().value;
            },

            getPoints: function () {
                return this._pointsCache;
            },

            _generateTimePoint: function () {
                return new Date().getTime();
            }           
        });


        return TrendlineBase;
    });