﻿define(['common/Enums', 'base/ObservableObject', 'common/Utilites', 'common/Appearence'],
    function (Enums, ObservableObject, Utilites, Appearence) {

        var events = Enums.TrendEvents,
            // options used for drawing
            defaults = {
                interval: 10000,
                max: new Date().getTime(),
                dateOffset: 2,
                dateWrap: true,
                reserveTickValuePadding: 2, //если padding TickValue=0, то текст наедет на тик
                labelFontName: 'Arial',
                labelFontSize: 18,
                labelTopPadding: 4,
                font: '12px Arial'
            };

        var XAxis = ObservableObject.extend({
            init: function (canvas, size, model, max) {
                this._super();
                this.canvas = canvas;
                this.size = size;
                this.width = size.width;
                this.model = model;
                this.options = $.extend(true, {}, defaults);
                this.options.max = max;
            },

            position: function (coords) {
                this.x = coords.x;
                this.y = coords.y;
                this.timeFormatY;
                this._updateHeight();
                this._positionTicks();
                this.eventTarget.fire(events.position);
            },

            getMin: function () {
                return this.options.max - this.model.get(Enums.ParameterRoles.INTERVAL);
            },

            getMajorTicks: function () {
                return this._majorTickCoords;
            },

            getMinorTicks: function () {
                return this._minorTickCoords;
            },

            getDimensions: function () {
                return {
                    x: this.x,
                    y: this.y,
                    width: this.width,
                    height: this.height
                }
            },

            draw: function () {
                this._updateHeight();                
                this.canvas.clearRect(this.getDimensions());
                this.drawAxis();
                this.drawArrow();
                this.drawTicks();
                this.drawLabel();
                this.eventTarget.fire(events.drawTicks);
            },

            redrawTicks: function () {               

                var tickSize = this._getTickSize(this.options.max),
                    h = tickSize.height,
                    minAxisThikness = this.model.get(Enums.ParameterRoles.X_AXIS_THICKNESS) / 2 +
                    this.model.get(Enums.ParameterRoles.X_MAJOR_TICK_HEIGHT) / 2 +
                     +this.model.get(Enums.ParameterRoles.X_MAJOR_TICK_PADDING_TOP);

                this.canvas.clearRect({
                    x: 0,
                    y: this.y + minAxisThikness,
                    width: this.width + this.options.reserveTickValuePadding,
                    height: h
                });
                this.timeFormatY = this.y + minAxisThikness;
                this.drawTicks();
                this.eventTarget.fire(events.drawTicks);
            },

            drawAxis: function () {
                var axis = {};
                // black rect - line
                axis.fillStyle = Appearence.color.toCssColor(this.model.get(Enums.ParameterRoles.X_AXIS_FILL_COLOR));
                axis.strokeStyle = Appearence.color.toCssColor(this.model.get(Enums.ParameterRoles.X_AXIS_STROKE_COLOR));
                axis.x = this.x;
                axis.y = this.y;

                axis.width = this.size.width - this.x;
                axis.height = this.model.get(Enums.ParameterRoles.X_AXIS_THICKNESS);
                this.canvas.drawRect(axis);
                axis = null;
            },

            drawLabel: function () {
                var text = this.model.get(Enums.ParameterRoles.X_AXIS_LABEL);
                if (!text) {
                    return;
                }
                var font = String.format('{0}px {1}', this.options.labelFontSize, this.options.labelFontName);
                this.canvas.setFont(font);
                var txtWidth = this.canvas.measureText(text);
                var xPos = (this.width / 2) - (txtWidth / 2);
                var yPos = this._getTickSize(this.options.max).height
                     + this.model.get(Enums.ParameterRoles.X_MAJOR_TICK_PADDING_TOP)
                    + this._getAxisThickness() / 2 + this._getLabelPadding();

                this.canvas.setFontStyle(font);
                this.canvas.fillText({
                    text: text,
                    x: this.x + xPos,
                    y: this.y + yPos,
                    maxWidth: this.width
                });
                this.canvas.setFontStyle(this.options.font);
                this.canvas.setFont(this.options.font);
            },

            _positionTicks: function () {
                this._positionMajorTicks();
                this._positionMinorTicks();
            },

            _positionMajorTicks: function () {
                var offset = this._calcMajorTickGaps(),
                    x = this.x,
                    y = this.y + this.model.get(Enums.ParameterRoles.X_AXIS_THICKNESS) / 2
                    - this.model.get(Enums.ParameterRoles.X_MAJOR_TICK_HEIGHT) / 2,
                    bbox, value;

                this._majorTickValuesBbox = [];
                this._majorTickCoords = [];
                var majorTickCount = this.model.getXMajorTickCount();
                //кол-во отрезков, поэтому +1 (первая засечка в начале координат)
                majorTickCount = majorTickCount == 0 ? 0 : majorTickCount + 1;

                for (var i = 0; i < majorTickCount; ++i) {
                    value = Utilites.GetIntervalValue(i / majorTickCount, this.getMin(), this.options.max);
                    bbox = this._positionMajorTickValue(value, x, y);
                    this._majorTickCoords.push([x, y]);
                    this._majorTickValuesBbox.push(bbox);
                    x += offset;
                }

                this._skipTickValues();
            },

            _positionMinorTicks: function () {
                var offset = this._calcMinorTickGaps(),
                 x = this.x,
                 y = this.y + this.model.get(Enums.ParameterRoles.X_AXIS_THICKNESS) / 2
                 - this.model.get(Enums.ParameterRoles.X_MINOR_TICK_HEIGHT) / 2,
                 bbox, value;
                this._minorTickValuesBbox = [];
                this._minorTickCoords = [];
                //Если задано число 5, то должно быть 5 вторичных интервалов (4 вторичных насечки). WI 10370
                var minorTickCount = this.model.get(Enums.ParameterRoles.X_MINOR_TICK_COUNT) - 1;
                var majorTickCount = this.model.getXMajorTickCount();

                var i, j;

                for (j = 0; j < majorTickCount; ++j) {
                    x += offset; //отступ от мажорного тика
                    for (i = 0; i < minorTickCount; ++i) {
                        this._minorTickCoords.push([x, y]);
                        x += offset;
                    }
                }
            },

            drawTicks: function () {
                this.drawMajorTicks();
                this.drawMinorTicks();
            },

            drawMajorTicks: function () {
                var value,
                    majorTickCount = this.model.getXMajorTickCount();

                //кол-во отрезков, поэтому +1 (первая засечка в начале координат)
                majorTickCount = majorTickCount == 0 ? 0 : majorTickCount + 1;

                for (var i = 0; i < majorTickCount; ++i) {
                    var x = this._majorTickCoords[i][0],
                        y = this._majorTickCoords[i][1];

                    this.drawMajorTick(x, y);
                    value = Utilites.GetIntervalValue(i / (majorTickCount - 1), this.getMin(), this.options.max);
                    if (this._majorTickValuesBbox[i].skip !== true) {
                        this.drawMajorTickValue(x, y, value, i);
                    }
                }
            },

            drawMinorTicks: function () {
                var value,
                    minorTickCount = this.model.get(Enums.ParameterRoles.X_MINOR_TICK_COUNT) - 1,
                    majorTickCount = this.model.getXMajorTickCount();

                var i, j;

                for (var i = 0; i < this._minorTickCoords.length; ++i) {
                    var x = this._minorTickCoords[i][0],
                        y = this._minorTickCoords[i][1];
                    this.drawMinorTick(x, y);
                }
            },

            drawMinorTick: function (x, y) {
                this.drawTick(x, y, this.model.getXMinorTickColor(),
                    this.model.getXMinorTickWidth(),
                    this.model.getXMinorTickHeight());
            },

            drawMajorTick: function (x, y) {
                this.drawTick(x, y, this.model.getXMajorTickColor(),
                    this.model.getXMajorTickWidth(),
                    this.model.getXMajorTickHeight());
            },

            drawTick: function (x, y, color, tickWidth, tickHeight) {
                var tick = {};
                tick.fillStyle = Appearence.color.toCssColor(color);
                // tick.strokeStyle = Appearence.color.toCssColor(this.options.color);
                tick.x = x;
                tick.y = y;
                tick.width = tickWidth;
                tick.height = tickHeight;

                this.canvas.clearRect(tick);
                this.canvas.drawRect(tick);
                tick = null;
            },

            drawMajorTickValue: function (x, y, value, i) {
                var dt = new Date(value),
                    bbox = this._majorTickValuesBbox[i];

                this.canvas.setFillStyle(Appearence.color.toCssColor(this.model.get(Enums.ParameterRoles.TEXT_COLOR)));
                if (this.options.dateWrap === false) {
                    this._drawTickSingleLine(bbox, dt);
                } else {
                    this._drawTickWithWrap(bbox, dt);
                }
            },

            _positionMajorTickValue: function (value, x, y) {
                var dt = new Date(value),
                    offset = this._getTickOffset(value),
                    size = this._getTickSize(value),
                    bbox;

                if (this.options.dateWrap === false) {
                    bbox = this._positionTickSingleLine(x, y, offset, dt);
                } else {
                    bbox = this._positionTickWithWrap(x, y, offset, dt, size);
                }

                return bbox;
            },

            _positionTickSingleLine: function (x, y, offset, dt) {
                var size = this._getTextSize(this.model.get(Enums.ParameterRoles.X_AXIS_FORMAT), dt),
                    offsetX = this.model.get(Enums.ParameterRoles.X_MAJOR_TICK_WIDTH) / 2 - size.width / 2;
                return {
                    x: x + offsetX,
                    y: y + offset.y,
                    width: size.width
                };
            },

            _positionTickWithWrap: function (x, y, offset, dt, size) {
                var rx, ry;
                timeFormat = this._getTimeFormat();
                if (timeFormat !== null) {
                    rx = x + this.model.get(Enums.ParameterRoles.X_MAJOR_TICK_WIDTH) / 2
                        - this._getTimeSize(dt).width / 2;
                    ry = y + offset.y;
                }

                var dateFormat = this._getDateFormat();
                if (dateFormat !== null) {
                    var dateOffsetX = this.model.get(Enums.ParameterRoles.X_MAJOR_TICK_WIDTH) / 2
                        - this._getDateSize(dt).width / 2;
                    rx < x + dateOffsetX ? rx = rx : rx = x + dateOffsetX;
                }

                return {
                    x: rx,
                    y: ry,
                    width: size.width
                }
            },

            _drawTickSingleLine: function (bbox, dt) {
                var text = String.format(this.model.get(Enums.ParameterRoles.X_AXIS_FORMAT), dt),
                    height = this.canvas.measureText(text).height;

                this.canvas.fillText({
                    text: text,
                    x: bbox.x,
                    y: bbox.y
                });
            },

            _drawTickWithWrap: function (bbox, dt) {
                var timeFormat = this._getTimeFormat(),
                    previousSize = bbox.y;
                if (timeFormat !== null) {
                    var timeValue = String.format(timeFormat, dt),
                        timeHeight = this.canvas.measureText(timeValue).height;

                    this.canvas.fillText({
                        text: timeValue,
                        x: bbox.x,
                        y: this.timeFormatY
                    });
                }

                var dateFormat = this._getDateFormat();
                if (dateFormat !== null) {
                    previousSize += this._getTimeSize(dt).height + this.options.dateOffset;
                    var dateValue = String.format(dateFormat, dt),
                        dateHeight = this.canvas.measureText(dateValue).height;

                    this.canvas.fillText({
                        text: dateValue,
                        x: bbox.x,
                        y: this.timeFormatY + this._getTimeSize(dt).height + this.options.dateOffset
                    });
                }
            },

            _getTickOffset: function (value) {
                var valueSize = this._getTickSize(value).width;
                return {
                    x: this.model.get(Enums.ParameterRoles.X_MAJOR_TICK_WIDTH) / 2 - valueSize / 2,
                    y: this.model.get(Enums.ParameterRoles.X_MAJOR_TICK_HEIGHT) / 2
                        + this.model.get(Enums.ParameterRoles.X_AXIS_THICKNESS) / 2
                        + this.model.get(Enums.ParameterRoles.X_MAJOR_TICK_PADDING_TOP)
                        + this.options.reserveTickValuePadding
                }
            },

            _skipTickValues: function () {
                // первое и последнее значение рисуем всегда
                // для любого другого можно выставлять skip в true
                var majorTickCount = this.model.getXMajorTickCount(),
                    pbbox = this._majorTickValuesBbox[0],
                    bbox;
                for (var i = 1; i < majorTickCount; ++i) {
                    bbox = this._majorTickValuesBbox[i];
                    if (bbox.x < pbbox.x + pbbox.width) {
                        bbox.skip = true;
                    } else {
                        pbbox = bbox;
                    }
                }

                var lbbox = this._majorTickValuesBbox[majorTickCount - 1],
                    plbbox = this._majorTickValuesBbox[majorTickCount - 2];
                if (plbbox && lbbox.x < plbbox.x + plbbox.width) {
                    plbbox.skip = true;
                }
            },

            drawArrow: function () {
                var startX = this.width - this.model.get(Enums.ParameterRoles.X_ARROW_WIDTH),
                    startY = this.y - this.model.get(Enums.ParameterRoles.X_ARROW_HEIGHT) / 2
                    + this.model.get(Enums.ParameterRoles.X_AXIS_THICKNESS) / 2;

                this.canvas.beginPath();
                this.canvas.moveTo({ x: startX, y: startY });
                this.canvas
                    .setStrokeStyle(Appearence.color.toCssColor(this.model.get(Enums.ParameterRoles.X_ARROW_STROKE_COLOR)))
                    .lineTo({ x: this.width, y: this.y + this.model.get(Enums.ParameterRoles.X_AXIS_THICKNESS) / 2 })
                    .lineTo({
                        x: startX, y: this.y + this.model.get(Enums.ParameterRoles.X_ARROW_HEIGHT) / 2
                            - this.model.get(Enums.ParameterRoles.X_AXIS_THICKNESS) / 2
                    })
                    .closePath()
                    .setFillStyle(Appearence.color.toCssColor(this.model.get(Enums.ParameterRoles.X_ARROW_FILL_COLOR)))
                    .fill()
                    .stroke();
            },

            _calcMajorTickGaps: function () {
                // учитываем смещение по X
                return (this.width - this.model.get(Enums.ParameterRoles.X_ARROW_MARGIN)
                    - this.model.get(Enums.ParameterRoles.X_ARROW_WIDTH) - this.x) / (this.model.getXMajorTickCount());
            },

            _calcMinorTickGaps: function () {
                return this._calcMajorTickGaps()
                    / (this.model.get(Enums.ParameterRoles.X_MINOR_TICK_COUNT));
            },

            // height of axis bounding region
            _updateHeight: function () {
                this.height = this._getTickLineSize() + this.model.get(Enums.ParameterRoles.X_MAJOR_TICK_PADDING_TOP)
                    + this._getAxisThickness() / 2 + this._getLabelSize()
                    + this.model.get(Enums.ParameterRoles.X_MAJOR_TICK_PADDING_BOTTOM);
            },

            _getTickLineSize: function () {
                if (this.model.getXMajorTickCount() > 0) {
                    return this._getTickSize(this.options.max).height
                }

                return 0;
            },

            _getLabelSize: function () {
                return this.model.get(Enums.ParameterRoles.X_AXIS_LABEL) ? this.options.labelFontSize + this.options.labelTopPadding : 0;
            },

            _getLabelPadding: function () {
                return this.model.get(Enums.ParameterRoles.X_AXIS_LABEL) ? this.options.labelTopPadding : 0;
            },

            _getAxisThickness: function () {
                return Math.max(this.model.get(Enums.ParameterRoles.X_AXIS_THICKNESS),
                    this.model.get(Enums.ParameterRoles.X_ARROW_HEIGHT), this.model.get(Enums.ParameterRoles.X_MAJOR_TICK_HEIGHT));
            },

            _getTickSize: function (value) {
                var dt = new Date(value),
                    timeSize = this._getTimeSize(dt),
                    dateSize = this._getDateSize(dt),
                    others = this.model.get(Enums.ParameterRoles.X_AXIS_FORMAT)
                        .replace(this._getTimeFormat(), '')
                        .replace(this._getDateFormat(), '')
                        .replace(/{0:/, '')
                        .replace('}', ''),
                    othersSize = this.canvas.measureText(others);

                if (this.options.dateWrap === false) {
                    height = timeSize.height === 0 ? dateSize.height : timeSize.height;
                    width = timeSize.width + othersSize + dateSize.width;
                } else {
                    width = timeSize.width > dateSize.width ? timeSize.width : dateSize.width;
                    width += othersSize;
                    height = timeSize.height + dateSize.height + this.options.dateOffset;
                }

                return {
                    width: width,
                    height: height
                };
            },

            _getTextSize: function (format, value) {
                if (format === null) {
                    return {
                        width: 0,
                        height: 0
                    };
                }

                var dateWidth = this.canvas.measureText(String.format(format, value));
                return {
                    width: dateWidth,
                    height: 12
                };
            },

            _getDateSize: function (value) {
                return this._getTextSize(this._getDateFormat(), value);
            },

            _getTimeSize: function (value) {
                return this._getTextSize(this._getTimeFormat(), value);
            },

            _getDateFormat: function () {
                return this._getFormat(this.model.get(Enums.ParameterRoles.X_AXIS_FORMAT), /[dMyE]/g);
            },

            _getTimeFormat: function () {
                return this._getFormat(this.model.get(Enums.ParameterRoles.X_AXIS_FORMAT), /[hHmsf]/g);
            },

            _getFormat: function (format, tokenRegExp) {
                var result = tokenRegExp.exec(format);
                if (result != null) {
                    var startIndex = result.index;
                    var endIndex = startIndex;
                    while (result != null) {
                        result = tokenRegExp.exec(format);
                        if (result != null) {
                            endIndex = result.index;
                        }
                    }

                    return format.substring(startIndex, endIndex + 1);
                }

                return null;
            },

            getAxisLineLength: function () {
                return (this.width - this.model.get(Enums.ParameterRoles.X_ARROW_MARGIN)
                    - this.model.get(Enums.ParameterRoles.X_ARROW_WIDTH) - this.x);
            }
        });

        return XAxis;
    });