﻿define(['common/Enums', 'base/ObservableObject', 'common/Utilites', 'common/Appearence'],
    function (Enums, ObservableObject, Utilites, Appearence) {

        var events = Enums.TrendEvents,
            // options used for drawing
            defaults = {             
                majorTicksCount: 5,
                format: '{0:n2}',
                defaultFormat: '{0:n2}',
                min: Number.POSITIVE_INFINITY,
                max: Number.NEGATIVE_INFINITY,
                labelFontName: 'Arial',
                labelFontSize: 18,
                labelRightPadding: 1,
                labelLeftPadding: 4,
                font: '12px Arial'
            };        

    var YAxis = ObservableObject.extend({

        // how much overall height will take yAxis
        heightUsage: 0.8,

        init: function (canvas, size, model) {
            this._super();
            this.canvas = canvas;
            this.size = size;
            this.height = size.height; // * this.heightUsage;            
            this.model = model;
            this.options = $.extend(true, {}, defaults);
            this.options.format = this.options.format ||
                defaults.defaultFormat;            
        },

        position: function (coords) {
            this.y = coords.y;
            this.x = coords.x;
            this.updateWidth();
            this.positionTicks();         
            this.eventTarget.fire(events.position);
        },

        update: function (settings) {
            if (settings) {
                this.options.min = +settings.min;
                this.options.max = +settings.max;
                this.options = $.extend(true, this.options, settings);
            }

             // TODO if bug will appear, move updateWidth call above $.extend
            this.updateWidth();
            this.options.format = this.options.format ||
               defaults.defaultFormat;
        },

        draw: function () {
            this.canvas.clearRect(this.getDimensions());
            this.drawAxis();
            this.drawArrow();
            this.drawTicks();
            this.drawLabel();
            this.eventTarget.fire(events.drawTicks);
        },

        drawAxis: function () {
            var axis = {};
            // black rect - line
            axis.fillStyle = Appearence.color.toCssColor(this.model.get(Enums.ParameterRoles.Y_AXIS_FILL_COLOR));
            axis.strokeStyle = Appearence.color.toCssColor(this.model.get(Enums.ParameterRoles.Y_AXIS_STROKE_COLOR));
            axis.x = this.getAxisX();
            axis.y = this.y;
            axis.width = this.model.get(Enums.ParameterRoles.Y_AXIS_THICKNESS);
            axis.height = this.height; // - this.y;
            this.canvas.drawRect(axis);
        },

        positionTicks: function () {
            this.positionMajorTicks();
            this.positionMinorTicks();
        },

            positionMajorTicks: function () {
            var offset = this.calcMajorTickGaps(),
                    x = this.getMinMajorTickX();
            y = this.height + this.y;


            this._majorTickCoords = [];
                //кол-во отрезков, поэтому +1 (первая засечка в начале координат)
                var majorTicksCount = this.options.majorTicksCount === 0 ?
                    0 : this.options.majorTicksCount + 1;

                for (var i = 0; i < majorTicksCount ; ++i) {
                this._majorTickCoords.push([x, y]);
                y -= offset;
            }
        },

            getMinMajorTickX: function () {
                return this.getAxisX() + this.model.get(Enums.ParameterRoles.Y_AXIS_THICKNESS) / 2
                    - this.model.get(Enums.ParameterRoles.Y_MAJOR_TICK_WIDTH) / 2
                    - this.model.get(Enums.ParameterRoles.Y_MAJOR_TICK_PADDING_LEFT)
                    - this.model.get(Enums.ParameterRoles.Y_MAJOR_TICK_PADDING_RIGHT);
            },

        drawMajorTicks: function () {
                //кол-во отрезков, поэтому +1 (первая засечка в начале координат)
                var majorTickCount = this.options.majorTicksCount === 0 ?
                     0 : this.options.majorTicksCount + 1;

                for (var i = 0; i < majorTickCount; ++i) {
                var x = this._majorTickCoords[i][0],
                    y = this._majorTickCoords[i][1];

                this.drawMajorTick(x, y);
                    value = Utilites.GetIntervalValue(i / (majorTickCount - 1), this.options.min, this.options.max);
                this.drawMajorTickValue(x, y, value);
            }
        },

        positionMinorTicks: function () {
            var offset = this.calcMinorTickGaps(),
                x = this.getAxisX() + this.model.get(Enums.ParameterRoles.Y_AXIS_THICKNESS) / 2
                - this.model.get(Enums.ParameterRoles.Y_MINOR_TICK_WIDTH) / 2,
                y = this.height + this.y,
                 //Если задано число 5, то должно быть 5 вторичных интервалов (4 вторичных насечки). WI 10370
                minorTickCount = this.model.get(Enums.ParameterRoles.Y_MINOR_TICK_COUNT) - 1;

            this._minorTickCoords = [];
            var i, j;

            for (j = 0; j < this.options.majorTicksCount; ++j) {
                y -= offset; //отступ от мажорного тика
                for (i = 0; i < minorTickCount; ++i) {
                    this._minorTickCoords.push([x, y]);
                    y -= offset;
                }
            }
        },

        drawMinorTicks: function () {
            var minorTickCount = this.model.get(Enums.ParameterRoles.Y_MINOR_TICK_COUNT) - 1;
            var majorTickCount = this.options.majorTicksCount;
            for (var i = 0; i < minorTickCount * majorTickCount; ++i) {
                var x = this._minorTickCoords[i][0],
                    y = this._minorTickCoords[i][1];

                this.drawMinorTick(x, y);              
            }
        },

        drawTicks: function () {
            this.drawMajorTicks();
            this.drawMinorTicks();
        },

        drawMajorTick: function (x, y) {          
            this.drawTick(x, y, this.model.getYMajorTickColor(), this.model.getYMajorTickWidth(),
                this.model.getYMajorTickHeight());
        },

        drawMinorTick: function (x, y) {
            this.drawTick(x, y, this.model.getYMinorTickColor(), this.model.getYMinorTickWidth(),
                this.model.getYMinorTickHeight());
        },

        drawTick: function (x, y, color, width, height) {
            var tick = {};
            tick.fillStyle = Appearence.color.toCssColor(color);
            // tick.strokeStyle = Appearence.color.toCssColor(this.options.color);
            tick.x = x;
            tick.y = y;
            tick.width = width;
            tick.height = height;

            this.canvas.clearRect(tick);
            this.canvas.drawRect(tick);
            tick = null;
        },

            drawMajorTickValue: function (x, y, value) {
            value = String.format(this.options.format, value);
            var tickSize = this.getTickSize(value),
                xOffset = this.model.get(Enums.ParameterRoles.Y_MAJOR_TICK_PADDING_LEFT)
                    + this.canvas.measureText(value),
                yOffset = - tickSize.height / 2,
                x = x - xOffset,
                y = y + yOffset,
                maxWidth = tickSize.width;

            this.canvas.setFillStyle(Appearence.color.toCssColor(this.model.get(Enums.ParameterRoles.TEXT_COLOR)));
            this.canvas.fillText({
                text: value,
                x: x,
                y: y,
                maxWidth: maxWidth
            });
        },

        drawArrow: function () {
            var startX = this.getAxisX() - this.model.get(Enums.ParameterRoles.Y_ARROW_WIDTH) / 2 +
                this.model.get(Enums.ParameterRoles.Y_AXIS_THICKNESS) / 2,
                        startY = this.y + this.model.get(Enums.ParameterRoles.Y_ARROW_HEIGHT);

            this.canvas.beginPath();
            this.canvas.moveTo({ x: startX, y: startY });
            this.canvas
                .setStrokeStyle(Appearence.color.toCssColor(this.model.get(Enums.ParameterRoles.Y_ARROW_STROKE_COLOR)))
                .lineTo({ x: this.getAxisX() + this.model.get(Enums.ParameterRoles.Y_AXIS_THICKNESS) / 2, y: this.y })
                .lineTo({
                    x: this.getAxisX() + this.model.get(Enums.ParameterRoles.Y_ARROW_WIDTH) / 2
                        - this.model.get(Enums.ParameterRoles.Y_AXIS_THICKNESS) / 2, y: startY
                })
                .closePath()
                .setFillStyle(Appearence.color.toCssColor(this.model.get(Enums.ParameterRoles.Y_ARROW_FILL_COLOR)))
                .fill()
                .stroke();
        },

        drawLabel: function () {
            var text = this.model.get(Enums.ParameterRoles.Y_AXIS_LABEL);
            if (!text) {
                return;
            }

            var font = String.format('{0}px {1}', this.options.labelFontSize, this.options.labelFontName);
            this.canvas.setFont(font);
            var txtWidth = this.canvas.measureText(text);
            var xPos = this.x + this.options.labelLeftPadding;
            var actualHeight = this.height - this.y;
            var yPos = (actualHeight / 2);

            this.canvas.setFontStyle(font);
            this.canvas.fillText({
                text: text,
                x: xPos,
                y: yPos,
                maxWidth: actualHeight,
                angle: -90
            });
            this.canvas.setFontStyle(this.options.font);
            this.canvas.setFont(this.options.font);
        },

        calcMajorTickGaps: function () {
            return (this.height - this.model.get(Enums.ParameterRoles.Y_ARROW_MARGIN) -
                    this.model.get(Enums.ParameterRoles.Y_ARROW_HEIGHT)) / (this.options.majorTicksCount);
                // - this.options.majorTicksCount * this.options.majorTick.height;
        },

        calcMinorTickGaps: function () {            
                return this.calcMajorTickGaps()
                    / (this.model.get(Enums.ParameterRoles.Y_MINOR_TICK_COUNT));
        },

        // width of axis bounding region
        updateWidth: function () {
            this.width = this.getTickSize().width
                + this.model.get(Enums.ParameterRoles.Y_MAJOR_TICK_PADDING_LEFT)
                + this.model.get(Enums.ParameterRoles.Y_MAJOR_TICK_PADDING_RIGHT)
                + this.getAxisMaxWidth() + this._getLabelSize();
        },

        _getLabelSize: function () {
            return this.model.get(Enums.ParameterRoles.Y_AXIS_LABEL) ? this.options.labelFontSize + this.options.labelRightPadding +
            this.options.labelLeftPadding : 0;
        },      

        getDimensions: function () {
            return {
                x: this.x,
                y: this.y,
                // не учитываем ширину меток на оси
                width: this.getAxisX(),
                height: this.height
            }
        },

        getMajorTicks: function () {
            return this._majorTickCoords;
        },

        getMinorTicks: function () {
            return this._minorTickCoords;
        },

        getAxisX: function () {
            return this.width - this.model.get(Enums.ParameterRoles.Y_AXIS_THICKNESS);
        },

        getAxisMaxWidth: function () {
            return Math.max(
                this.model.get(Enums.ParameterRoles.Y_AXIS_THICKNESS),
                this.model.get(Enums.ParameterRoles.Y_ARROW_WIDTH),
                this.model.get(Enums.ParameterRoles.Y_MINOR_TICK_WIDTH)
            );
        },

        getTickSize: function () {
            var minLen = String.format(this.options.format, this.options.min).length,
                maxLen = String.format(this.options.format, this.options.max).length,             
                toMeasure = minLen > maxLen ? this.options.min : this.options.max;
            return {
                width: this.canvas.measureText(String.format(this.options.format, toMeasure)),
                height: this.canvas.getFontSize()
            };
        },

        getTickMaxSize: function () {
            },

            getAxisLineLength: function () {
                return (this.height - this.y - this.model.get(Enums.ParameterRoles.Y_ARROW_MARGIN) -
                    this.model.get(Enums.ParameterRoles.Y_ARROW_HEIGHT));
            },

            getYWithoutArrow: function () {
                return this.model.get(Enums.ParameterRoles.Y_MINOR_TICK_WIDTH) / 2 +
                            this.model.get(Enums.ParameterRoles.Y_ARROW_MARGIN) +
                            this.model.get(Enums.ParameterRoles.Y_ARROW_HEIGHT) + this.y;
            },

            getMajorTickCount: function () {
                return this.options.majorTicksCount;
        }

    });

    return YAxis;
});