/** @license MIT License (c) copyright 2013 original authors */
/**
 * String polyfill / shims
 *
 * @author Brian Cavalier
 * @author John Hann
 *
 * Adds str.trim(), str.trimRight(), and str.trimLeft()
 *
 * Note: we don't bother trimming all possible ES5 white-space characters.
 * If you truly need strict ES5 whitespace compliance in all browsers,
 * create your own trim function.
 * from http://perfectionkills.com/whitespace-deviations/
 * '\x09-\x0D\x20\xA0\u1680\u180E\u2000-\u200A\u202F\u205F\u3000\u2028\u2029'
 */
(function (define) {
define(function (require) {
"use strict";

	var base = require('./lib/_base');

	var proto = String.prototype,
		featureMap,
		has,
		toString;

	featureMap = {
		'string-trim': 'trim',
		'string-trimleft': 'trimLeft',
		'string-trimright': 'trimRight'
	};

	function checkFeature (feature) {
		var prop = featureMap[feature];
		return base.isFunction(proto[prop]);
	}

	function neg () { return false; }

	has = checkFeature;

	// compressibility helper
	function remove (str, rx) {
		return str.replace(rx, '');
	}

	toString = base.createCaster(String, 'String');

	var trimRightRx, trimLeftRx;

	trimRightRx = /\s+$/;
	trimLeftRx = /^\s+/;

	function checkShims () {
		if (!has('string-trim')) {
			proto.trim = function trim () {
				return remove(remove(toString(this), trimLeftRx), trimRightRx);
			};
		}

		if (!has('string-trimleft')) {
			proto.trimLeft = function trimLeft () {
				return remove(toString(this), trimLeftRx);
			};
		}

		if (!has('string-trimright')) {
			proto.trimRight = function trimRight () {
				return remove(toString(this), trimRightRx);
			};
		}

	}

	checkShims();

	return {
		setWhitespaceChars: function (wsc) {
			trimRightRx = new RegExp(wsc + '$');
			trimLeftRx = new RegExp('^' + wsc);
			// fail all has() checks and check shims again
			has = neg;
			checkShims();
		}
	};

});
}(
	typeof define == 'function' && define.amd
		? define
		: function (factory) { module.exports = factory(require); }
));
