﻿// <copyright file="Toolbar.js" company="ИнСАТ">
// ИнСАТ, 2014
// </copyright>
// 

define(['base/EventTarget', 'common/Enums', 'common/Appearence',
    'controls/Journal/modules/toolbar/ToolbarButtons', 'core/L10n'],
    function(EventTarget, Enums, Appearence, ToolbarButtons, L10n) {
        var placeholder = 'toolbar',
            stateKey = 'state',
            propertyKey = 'property',
            actionKey = 'action',
            TOGGLE_BUTTON_STATE_CHANGED = 'ToggleButtonStateChanged',
            BUTTON_CLICKED = 'ButtonClicked',
            ACTION_CALLED = 'ActionCalled',
            types = {
                toggle: 'toggle',
                simple: 'simple'
            },
            l10Prefix = "journalToolbar.",
            toggleBtnl10 = "property",
            ToolbarProto = {
                _model: {},
                view: {
                    format: '<div class="journal-toolbar"></div>'
                }
            },
            toggleButtonProto = $$({
                _model: {
                    state: false,
                    class: function() {
                        if (this.state) {
                            return this.onClass;
                        } else {
                            return this.offClass;
                        }
                    },
                    urlOn: null,
                    urlOff: null,
                    url: function() {
                        if (this.state) {
                            return this.urlOn;
                        } else {
                            return this.urlOff;
                        }
                    },

                    title: '',
                    property: null
                },
                view: {
                    format:
                        '<button data-bind="id=id, class=class, title=title"><img data-bind="src=url"></img></button>'
                },
                _controller: {
                    'click &': function() {
                        this.toggle();
                    }
                },
                toggle: function() {
                    var newState = !this._model.get(stateKey);
                    this.updateTitle(newState);
                    this._model.set({ state: newState }, { silent: true });
                    this.refresh();
                    this.trigger(TOGGLE_BUTTON_STATE_CHANGED,
                        {
                            property: this._model.get(propertyKey),
                            newState: newState
                        });
                },
                setState: function(newState) {
                    this._model.set({ state: newState }, { silent: true });
                    this.updateTitle(newState);
                    this.refresh();
                },
                updateTitle: function(newState) {
                    this._model.set({
                        title: L10n.get(l10Prefix + this._model.get(toggleBtnl10) + (newState ? 'On' : 'Off'))
                    });
                },
                refresh: function() {
                    this.view.sync();
                }
            }),
            buttonProto = $$({
                _model: {
                    state: false,
                    class: 'button',
                    imageUrl: null,
                    title: ''
                },
                view: {
                    format:
                        '<button data-bind="id=id, class=class, title=title"><img data-bind="src=imageUrl"></img></button>'
                },
                _controller: {
                    'click &': function() {
                        this.click();
                    }
                },
                click: function() {
                    this.trigger(BUTTON_CLICKED,
                        {
                            action: this._model.get(actionKey)
                        });
                }
            }),
            groupProto = $$({
                _model: {
                    id: null,
                    class: null
                },

                view: {
                    format: '<div data-bind="id=id, class=class"></div>'
                }
            });

        var Toolbar = Class.extend({
            init: function(journalModel) {
                this.eventTarget = new EventTarget();
                this.model = journalModel;
                this.model.subscribePropertyChanged(this.onModelPropertyChanged, this);
                this.buttonSpecs = $.extend(true, {}, ToolbarButtons);
                this.toggleButtons = {}; //PropName: {toggleButton}
                this.toolbar = null;
                this._onToggleButtonClickWrapper = this._onToggleButtonClick.bind(this);
                this._onButtonClickWrapper = this._onButtonClick.bind(this);
                this._createToolbar();
            },

            stylize: function() {

            },

            render: function(container) {
                container.children(String.format('[placeholder="{0}"]', placeholder)).append(this.toolbar.view.$());
                this._onIsEnabledChanged();
                this._onShowToolbarChanged();
            },

            afterRender: function(options) {

            },

            _createToolbar: function() {

                this.toolbar = $$(ToolbarProto);
                var i;
                for (i in this.buttonSpecs.groups) {
                    var group = this._createGroup(this.buttonSpecs.groups[i]);
                    this.toolbar.append(group);
                }

                var spec;
                for (i in this.buttonSpecs.buttons) {
                    spec = this.buttonSpecs.buttons[i];
                    var newObj = null;

                    if (spec.type === types.toggle) {
                        newObj = this._createToggleButton(spec);
                        newObj.setState(this.model.get(spec[propertyKey]));
                        newObj.bind(TOGGLE_BUTTON_STATE_CHANGED, this._onToggleButtonClickWrapper);
                        this.toggleButtons[spec[propertyKey]] = newObj;
                    } else if (spec.type === types.simple) {
                        newObj = this._createSimpleButton(spec);
                        newObj.bind(BUTTON_CLICKED, this._onButtonClickWrapper);
                    }

                    if (spec.groupId) {
                        this.toolbar.append(newObj, 'div[id=' + spec.groupId + ']');
                    } else {
                        this.toolbar.append(newObj);
                    }
                }
            },

            _createToggleButton: function(spec) {
                var ext = {
                    _model:
                    {
                        id: spec.id,
                        onText: spec.onText,
                        offText: spec.offText,
                        onClass: spec.onClass,
                        offClass: spec.offClass,
                        state: false,
                        args: spec.args,
                        property: spec.property,
                        action: spec.action,
                        title: ''
                    }
                };
                if (spec.image) {
                    ext._model.urlOn = spec.image.urlOn;
                    ext._model.urlOff = spec.image.urlOff;
                }

                return $$(toggleButtonProto, ext);
            },

            _createSimpleButton: function(spec) {
                return $$(buttonProto,
                    {
                        _model:
                        {
                            id: spec.id,
                            content: spec.text,
                            class: spec.class,
                            imageUrl: spec.imageUrl,
                            args: spec.args,
                            action: spec.action,
                            title: L10n.get(l10Prefix + spec.action)
                        }
                    });
            },

            _createGroup: function(spec) {
                return $$(groupProto,
                    {
                        _model: {
                            id: spec.id,
                            class: spec.class
                        }
                    });
            },

            onModelPropertyChanged: function(event) {
                if (this.toggleButtons[event.property] !== undefined) {
                    this.toggleButtons[event.property].setState(this.model.get(event.property));
                }

                switch (event.property) {
                case Enums.ParameterRoles.IS_ENABLED:
                    this._onIsEnabledChanged();
                    break;
                case Enums.ParameterRoles.SHOW_TOOLBAR:
                    this._onShowToolbarChanged();
                    break;
                }
            },

            _onShowToolbarChanged: function() {
                Appearence.applyVisibility(this.toolbar.view.$(),
                    this.model.get(Enums.ParameterRoles.SHOW_TOOLBAR));
            },

            _onIsEnabledChanged: function() {
                var isEnabled = this.model.get(Enums.ParameterRoles.IS_ENABLED);
                this.toolbar.view.$('button').prop("disabled", !isEnabled);
            },

            _onToggleButtonClick: function(event, args) {
                this.model.set(args.property, args.newState);
            },

            _onButtonClick: function(event, args) {
                this.fireActionCalled(args.action);
            },
            subscribeActionCalled: function(handler, context) {
                this.eventTarget.addListener(ACTION_CALLED, handler, context);
            },
            fireActionCalled: function(action) {
                this.eventTarget.fire({
                    type: ACTION_CALLED,
                    target: this,
                    action: action
                });
            },
            unsubscribeActionCalled: function(handler) {
                this.eventTarget.removeListener(ACTION_CALLED, handler);

            },
            destroy: function() {
                this.toolbar.view.$().remove();
                this.toolbar = null;
            }
        });

        return Toolbar;
    });