﻿// <copyright file="TrendVariablesManager.js" company="ИнСАТ">
// ИнСАТ, 2014
// </copyright>
// 


/** 
	* @class TrendVariablesManager
	* @classdesc Represents an object, which holds and manages state of "server variable" abstraction.
	
	*/
define(['server/VariablesManager', 'common/Enums', 'common/Error'],
    function (VariablesManager, Enums, Error) {

        var VAR_NOT_FOUND = 'GetArchiveItems: Variable {0} not found in archive. Status code {1}';
        var TrendVariablesManager = VariablesManager.extend({
            init: function (options) {
                this._super(options);
                this.className = 'TrendVariablesManager';
                this.variables.get.archive = {};
                this.variables.get.archiveInfo = {
                    minStartTime: 0,
                    maxEndTime: 0
                };
                this.requestParamsCache = [];
                this.requestId = 0;
            },

            _addGetVariableToSubscription: function (variable) {
                this._super(variable);
                variable.timestampsToReturn = Enums.timeStampReturnType.source;
                variable.queueSize = 0;
            },
            _updateVariable: function (variable, record) { //в record теперь массив с точками
                _.forEach(record, function (v) {
                    delete v.clientHandle;
                    if (!v.statusCode) {
                        v.statusCode = 0;
                    }
                });
                var oldvalue = variable.value;
                this.firePropertyChanged(variable.itemId + variable.path, oldvalue, record);
            },

            updateArchiveInfo: function () {
                if (!this._hasGetVariables()) {
                    return Promise.resolve();
                }
                this.requestParamsCache = this._getArchiveParams(this.variables.get.protocol);
                return this._updateArchiveInfo();
            },

            restartDataUpdate: function () {
                if (!this._hasGetVariables()) {
                    return Promise.resolve();
                }
                return this.dataAdapter.refresh();
            },

            getServerTime: function () {
                return this.serverStateManager.get(this.stateVars.serverTime);
            },

            initArchive: function () {
                if (!this._hasGetVariables()) {
                    return Promise.resolve();
                }

                var ctx = this;
                return this.dataAdapter.getArchiveItems(this.variables.get.protocol).then(function (items) {
                    var i;
                    for (i = 0; i < items.length; i++) {
                        if (items[i].statusCode !== undefined && items[i].statusCode !== 0) {
                            continue;
                        }

                        ctx.variables.get.protocol[i].archiveItemId = items[i].archiveItemId;
                        ctx.variables.get.archive[items[i].archiveItemId] = ctx.variables.get.protocol[i];
                    }
                    ctx.requestParamsCache = items.slice(0);
                });
            },

            getArchiveInfo: function () {
                return this.variables.get.archiveInfo;
            },
            getPensInfo: function () {
                return this.variables.get.protocol;
            },
            _getArchiveParams: function (variables) {
                return variables.map(function (variable) {
                    return {
                        archiveItemId: variable.itemId,
                        continuationPoint: 0
                    };
                });
            },

            _removeVariable: function (variable, removedVariable) {
                this._super(variable, removedVariable);
                removedVariable.usesCounts--;
            },

            _canDeleteSubscription: function () {
                var i;
                for (i = 0; i < this.variables.get.protocol.length; i++) {
                    if (this.variables.get.protocol[i].usesCounts !== 0) {
                        return false;
                    }
                }

                return true;
            },

            _updateArchiveInfo: function () {
                var params = {
                    startTime: 0,
                    endTime: 0,
                    returnBounds: false,
                    returnFirstLastValue: true
                }
                var ctx = this;

                return this.dataAdapter.getArchiveData(params, this.variables.get.protocol).then(function (items) {

                    var min = Number.POSITIVE_INFINITY,
                        max = Number.NEGATIVE_INFINITY;

                    _.map(items, function (item) {
                        if (item.statusCode !== undefined && item.statusCode !== 0) {
                            var i = _.indexOf(items, item);
                            ctx._logVarNotFound(ctx.variables.get.protocol[i][ctx.keyFieldName], item);
                        } else if (!item.values || item.values.length < 2) {
                            Error.info(String.format('Cannot get archive for {0}. Status code {1}',
                                item.archiveItemId, item.statusCode || 0));
                        } else {
                            // мы знаем, что в ответ на запрос придет ровно два значения
                            // причем в порядке возрастания
                            // поэтому можно схитрить
                            if (+item.values[0][0] < min) {
                                min = +item.values[0][0];
                            }
                            if (+item.values[1][0] > max) {
                                max = +item.values[1][0];
                            }
                        }
                    });

                    ctx.variables.get.archiveInfo.maxEndTime = max;
                    ctx.variables.get.archiveInfo.minStartTime = min;

                });
            },
            getAsyncArchiveData: function (startTime, endTime, resampleInterval, itemIdToGraphId) {
                if (!this._hasGetVariables()) {
                    return Promise.resolve();
                }
                var subscriptionId = this.serverStateManager.get(this.stateVars.SubscriptionId);
                var params = {
                    startTime: startTime,
                    endTime: endTime,
                    resampleInterval: resampleInterval,
                    subscriptionId: subscriptionId,
                    requestId: this.requestId
                };
                var ctx = this;
                this.requestId++;
                if (this.variables.get.protocol.length === 0) {
                    return Promise.resolve();
                }
                var storage = {};
                for (var i = 0; i < this.variables.get.protocol.length; i++) {
                    this.variables.get.protocol[i].continuationPoint = 0;
                    storage[this.variables.get.protocol[i].archiveItemId] = [];
                }
                return this._getAsyncArchiveData(params, this.variables.get.protocol, storage, itemIdToGraphId);
            },

            getArchiveData: function (startTime, endTime, resampleInterval, itemIdToGraphId) {
                if (!this._hasGetVariables()) {
                    return Promise.resolve();
                }

                var params = {
                    startTime: startTime,
                    endTime: endTime,
                    resampleInterval: resampleInterval
                };

                if (this.variables.get.protocol.length === 0) {
                    return Promise.resolve();
                }
                for (var i = 0; i < this.variables.get.protocol.length; i++) {
                    this.variables.get.protocol[i].continuationPoint = 0;
                }
                var storage = [];
                for (var i = 0; i < Object.keys(itemIdToGraphId).length; i++) {
                    storage.push({ data: [] });
                }
                return this._getArchiveData(params, this.variables.get.protocol, storage, itemIdToGraphId);
            },

            _getArchiveData: function (params, data, storage, itemIdToGraphId) {
                var ctx = this;
                params.startTime = params.startTime ? Math.floor(params.startTime) : params.startTime;
                params.endTime = params.endTime ? Math.ceil(params.endTime) : params.endTime;
                return this.dataAdapter.getArchiveData(params, data).then(function (items) {
                    var i, nextRequestData = [];
                    for (i = 0; i < items.length; i++) {
                        if (items[i].statusCode !== undefined && items[i].statusCode !== 0) {
                            this._logVarNotFound(ctx.variables.get.protocol[i][ctx.keyFieldName], items[i]);
                            continue;
                        }

                        if (items[i].areMoreValues === true && items[i].continuationPoint !== undefined) {
                            //получили не все данные
                            data[i].continuationPoint = items[i].continuationPoint;
                            nextRequestData.push(data[i]);
                        }

                        var values = items[i].values;
                        if (itemIdToGraphId != null) {
                            var idx = itemIdToGraphId[data[i].itemId + data[i].path];
                            if (idx != undefined) {
                                storage[idx].data = storage[i].data.concat(values);
                                continue;
                            }
                        }
                    }

                    data = null;
                    if (nextRequestData.length > 0) {
                        return ctx._getArchiveData(params, nextRequestData, storage, itemIdToGraphId);
                    } else {
                        return storage;
                    }
                });
            },

            _getAsyncArchiveData: function (params, data, storage, itemIdToGraphId) {
                var ctx = this;
                params.startTime = params.startTime ? Math.floor(params.startTime) : params.startTime;
                params.endTime = params.endTime ? Math.ceil(params.endTime) : params.endTime;
                return this.dataAdapter.getAsyncArchiveData(params, data).then(function (items) {
                    for (var i = 0; i < items.length; i++) {
                        if (items[i].statusCode !== undefined && items[i].statusCode !== 0) {
                            ctx._logVarNotFound(ctx.variables.get.protocol[i][ctx.keyFieldName], items[i]);
                            continue;
                        }
                        storage[items[i].archiveItemId] = storage[items[i].archiveItemId].concat(values);
                    }

                    var result = [];
                    for (var key in items) {
                        if (key != "0") //архива у пера нет
                            result[itemIdToGraphId[ctx.variables.get.archive[key].itemId]] = items[key];
                    }
                    return result;
                });
            },

            _hasGetVariables: function () {
                return this.variables.get.protocol.length > 0;
            },

            _logVarNotFound: function (id, item) {
                Error.info(String.format(VAR_NOT_FOUND, id, item.statusCode || 0));
            },
            _getProperty: function (id, isCopy) {
                if (this.variables.get.schema[id] !== undefined) {
                    return this._getPropertyValue(this.variables.get.schema[id], isCopy);
                } else if (this.variables.set.schema[id] !== undefined) {
                    return this._getPropertyValue(this.variables.set.schema[id], isCopy);
                }
                return "";
            },
            _getAllSucceed: function (recs) {
                var ctx = this;
                var arrayByClientHandle = _.groupBy(recs, 'clientHandle');
                _.forOwn(arrayByClientHandle, function (value, key) {
                    var variable = ctx.variables.get.callback[value[0].clientHandle];
                    if (variable !== undefined) {
                        ctx._updateVariable(variable, value);
                    } else {
                        console.warn('GetAllSucceed: Requested variable not found!!!!');
                    }
                });
                return Promise.resolve();
            },
            getKey: function (variable) {
                if (variable[this.keyFieldName])
                    return variable[this.keyFieldName] + variable[this.pathFieldName];
                else
                    return variable[this.pathFieldName];
            },
            unsubscribe: function (vars) {
                var ctx = this;
                return new Promise(function (resolve, reject) {
                    var i;
                    for (i in ctx.variables.set.schema) {
                        var existSetVariable = ctx.variables.get.schema[i];
                        if (existSetVariable && existSetVariable.usesCounts === 1) {
                            delete i;
                        } else {
                            existSetVariable.usesCounts--;
                        }
                    }
                    var monitoredItemsIds = [];
                    for (i in ctx.variables.get.schema) {
                        var removedVariable = ctx.variables.get.schema[i];
                        if (removedVariable.usesCounts === 1) {
                            removedVariable.value = undefined;
                            if (removedVariable.monitoredItemId !== undefined) {
                                monitoredItemsIds.push(removedVariable.monitoredItemId);
                            }

                            ctx._removeVariable(removedVariable, removedVariable);

                        } else {
                            //console.log('Decrease counter');
                            removedVariable.usesCounts--;
                            //console.log(removedVariable);
                        }
                    }

                    if (monitoredItemsIds.length > 0) {
                        ctx._unsubscribeVars(monitoredItemsIds).then(function () {
                            resolve();
                        },
                            function () {
                                reject('Unsubscribe Vars rejected');
                            });
                    }
                    resolve();
                });
            },
            subscribePen: function () {
                var ctx = this;
                this._addMonitoredData({ data: { items: [] } }).then(function (result) {
                    if (!result.data.items) {
                        reject('CreateMonitoredDataItems �� ������, ������ ������ ������: ' + result.data);
                    }
                    var items = result.data.items;
                    for (var i = 0; i < items.length; i++) {
                        if (ctx.variables.get.callback[items[i].clientHandle]) {
                            ctx.variables.get.callback[items[i].clientHandle].monitoredItemId = items[i].monitoredItemId;
                        }
                    }
                });
            },
            unsubscribePen: function (vars) {
                var ctx = this;
                return new Promise(function (resolve, reject) {
                    var i;
                    //get - удаляем + отписываемся
                    var monitoredItemsIds = [];
                    if (vars.getValues !== undefined) {
                        for (i = 0; i < vars.getValues.length; i++) {
                            var variable = vars.getValues[i];
                            var removedVariable = ctx.variables.get.schema[ctx.getKey(variable)];
                            if (removedVariable.usesCounts === 1) {
                                removedVariable.value = undefined;
                                if (removedVariable.monitoredItemId !== undefined) {
                                    monitoredItemsIds.push(removedVariable.monitoredItemId);
                                }
                                ctx._removeVariable(variable, removedVariable);
                            } else {
                                removedVariable.usesCounts--;
                            }
                        }
                    }

                    if (monitoredItemsIds.length > 0) {
                        ctx._unsubscribeVars(monitoredItemsIds).then(function () {
                            resolve();
                        },
                            function () {
                                reject('Unsubscribe Vars rejected');
                            });
                    }
                });
            }
        });
        return TrendVariablesManager;
    });